# towers_of_hanoi.py

"""
The Towers of Hanoi for wxPython
by S. Foster, Jan. 2004
Link : https://code.activestate.com/recipes/577511-hanoi-towers-solver-wxpython/

Based on wxHanoi.cpp
By Martin Bernreuther
"""

import wx

COLOURS = ['RED', 'CORAL', 'YELLOW', 'GREEN', 'BLUE', 'MAGENTA', 'PURPLE']

# class wxHanoiDisc
# class wxHanoiFrame
# class wxHanoiApp

#---------------------------------------------------------------------------

class wxHanoiDisc:
    """
    ...
    """
    def __init__(self, n, width, height):

        #------------
        
        self.width = (n + 1) * width
        self.height = height
        self.n = n
        self.brush = wx.Brush(wx.Colour(COLOURS[n % len(COLOURS)]))

#---------------------------------------------------------------------------
        
class wxHanoiFrame(wx.Frame):
    """
    ...
    """
    def __init__(self, title, size):
        style=wx.DEFAULT_FRAME_STYLE ^ wx.RESIZE_BORDER ^ wx.MAXIMIZE_BOX
        wx.Frame.__init__(self, None, -1,
                          title, size=size,
                          style=style)

        #------------
        
        self.SetIcon(wx.Icon(".\icons\wxwin.ico"))

        #------------

        self.sleepTime = 3
        self.numDiscs = 4

        self.pen = wx.Pen(wx.Colour('BLACK'), 1, wx.SOLID)

        #------------
        
        self.CreateMenu()
        self.CreateStatusBar()        
        self.Initialize()
        self.BindEvents()

        #------------
        
        self.CenterOnScreen(wx.BOTH)
        self.Show(True)
        
    #-----------------------------------------------------------------------
        
    def CreateMenu(self):
        def MenuCallback(menu, item, callback):
            id = wx.NewIdRef()
            menu.Append(id, item)
            self.Bind(wx.EVT_MENU, callback, id)

        menuBar = wx.MenuBar()

        menuFile = wx.Menu()
        MenuCallback(menuFile, 'E&xit...\tCtrl-X', self.OnQuit)
        menuBar.Append(menuFile, '&File')

        menuPlay = wx.Menu()
        MenuCallback(menuPlay, '&Number of Discs...', self.OnInpNumDiscs)
        MenuCallback(menuPlay, '&Time between Moves...', self.OnInpSleepTime)
        menuPlay.AppendSeparator()
        MenuCallback(menuPlay, '&Play', self.OnPlay)
        MenuCallback(menuPlay, '&Reset', self.OnReset)
        menuBar.Append(menuPlay, '&Play')

        menuHelp = wx.Menu()
        MenuCallback(menuHelp, '&About...', self.OnAbout)
        menuBar.Append(menuHelp, '&Help')

        self.SetMenuBar(menuBar)

       
    def Initialize(self):
        self.width, self.height = self.GetClientSize()
        maxwidth = self.width / 3
        w = self.width / 6
        self.xpos = [i * w for i in [1, 3, 5]]

        height = self.height / self.numDiscs
        width = maxwidth / self.numDiscs
        if height > width:
            height = width
        self.discheight = height
        self.discwidthfac = width
   
        discs = list(range(self.numDiscs))
        discs.reverse()

        self.pegs = [[wxHanoiDisc(i, width, height) for i in discs], [], []]
        self.moves = 0

        width, height = self.GetClientSize()
        self.Draw(wx.ClientDC(self), width, height)


    def BindEvents(self):
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)
        self.Bind(wx.EVT_PAINT, self.OnPaint)

        
    def OnQuit(self, event):
        self.Close(True)


    def OnCloseWindow(self, event):
        self.Destroy()


    def OnAbout(self, event):
        wx.MessageBox(__doc__, 'About wxHanoi', wx.OK|wx.ICON_INFORMATION, self)


    def OnInpNumDiscs(self, _event):
        self.numDiscs = wx.GetNumberFromUser('', 'Discs:', 'Number of Discs',
                                             self.numDiscs, 1, 25, self)
        if self.numDiscs == -1:
            self.SetStatusText("Invalid number entered or dialog cancelled.")
        else:
            self.Initialize()

        
    def OnInpSleepTime(self, event):
        self.sleepTime = wx.GetNumberFromUser('', 'Wait [sec/10]:', 'Time between Moves',
                                              self.sleepTime, 0, 10, self)
        if self.sleepTime == -1:
            self.SetStatusText("Invalid number entered or dialog cancelled.")
        else:
            self.Initialize()


    def OnPlay(self, event):
        self.Initialize()
        self.SetStatusText('Playing')
        self.Move(0, 2, 1, self.numDiscs)
        self.SetStatusText('Finished: %s Moves' % self.moves)

    
    def OnReset(self, event):
        self.Initialize()


    def DrawDisc(self, dc, disc, x, y):
        assert x <= 2
        dc.SetPen(self.pen)
        dc.SetBrush(disc.brush)
        dc.DrawRectangle(int(self.xpos[x] - (disc.width / 2)),
                         int(self.height - (y * self.discheight)),
                         int(disc.width),
                         int(disc.height))


    def Draw(self, dc, width, height):
        mdc = wx.MemoryDC()
        mdc.SelectObject(wx.Bitmap(width, height))
        for x, peg in enumerate(self.pegs):
            for y, disc in enumerate(peg):
                self.DrawDisc(mdc, disc, x, y+1)
        dc.Blit(0, 0, width, height, mdc, 0, 0)

        
    def OnPaint(self, event):
        width, height = self.GetClientSize()
        self.Draw(wx.PaintDC(self), width, height)


    def MoveDisc(self, src, dst):
        disc = self.pegs[src].pop()
        self.pegs[dst].append(disc)
        self.moves += 1
        self.SetStatusText('Move %s' % self.moves)
        width, height = self.GetClientSize()
        self.Draw(wx.ClientDC(self), width, height)
        wx.MilliSleep(10 * self.sleepTime)


    def Move(self, src, dst, temp, n):
        if n == 1:
            self.MoveDisc(src, dst)
        else:
            self.Move(src, temp, dst, n-1)
            self.MoveDisc(src, dst)
            self.Move(temp, dst, src, n-1)

#---------------------------------------------------------------------------
            
class wxHanoiApp(wx.App):
    """
    ...
    """
    def OnInit(self):

        #------------
        
        self.frame = wxHanoiFrame('Towers of Hanoi',
                                  wx.Size(450, 350))
        self.SetTopWindow(self.frame)

        return True

#---------------------------------------------------------------------------
    
def main():
    app = wxHanoiApp()
    app.MainLoop()

#---------------------------------------------------------------------------
    
if __name__ == '__main__':
    main()
    
