# tetris.py

"""
ZetCode wxPython tutorial.

This is Tetris game clone in wxPython.

Author : Jan Bodnar
Website : www.zetcode.com
Link : http://zetcode.com/wxpython/thetetrisgame/
Last modified : May 2018
"""

import wx
import random

# class Tetris
# class Board
# class Tetrominoes
# class Shape

#---------------------------------------------------------------------------

class Tetris(wx.Frame):
    """
    ...
    """
    def __init__(self, parent):
        style=wx.DEFAULT_FRAME_STYLE ^ wx.RESIZE_BORDER ^ wx.MAXIMIZE_BOX
        wx.Frame.__init__(self, parent, size=(220, 420), style=style)
        
        self.initFrame()
        
    #-----------------------------------------------------------------------
        
    def initFrame(self):
        """
        ...
        """
        
        self.statusbar = self.CreateStatusBar()
        self.statusbar.SetStatusText('0')
        self.board = Board(self)
        self.board.SetFocus()
        self.board.start()

        #------------

        # Creating the menubar.
        menu_bar = wx.MenuBar()
        
        # Setting up the menu.
        file_menu = wx.Menu()

        # wx.ID_ABOUT and wx.ID_EXIT are standard IDs provided by wxWidgets.
        item = file_menu.Append(wx.ID_ABOUT, "&About", "Information about Tetris.")
        self.Bind(wx.EVT_MENU, self.OnAbout, item)
        file_menu.AppendSeparator()
        item = file_menu.Append(wx.ID_EXIT, "E&xit", "Exit Tetris.")
        self.Bind(wx.EVT_MENU, self.OnQuit, item)

        # Create the "Help" top menu.
        help_menu = wx.Menu()
        item = help_menu.Append(wx.ID_HELP, "&Shortcuts", "Shortcuts for Tetris.")
        self.Bind(wx.EVT_MENU, self.OnHelp, item)

        # Adding the "file_menu" to the menu bar.
        menu_bar.Append(file_menu, "&File")
        menu_bar.Append(help_menu, "&Help")
        
        # Adding the menu bar to the frame content.
        self.SetMenuBar(menu_bar)
        
        #------------
        
        self.SetIcon(wx.Icon(".\icons\wxwin.ico"))
        self.SetTitle("Tetris")
        
        #------------
        
        self.CenterOnScreen(wx.BOTH)

    
    def OnQuit(self, event):
        self.Destroy()

        
    def OnAbout(self, event):
        dlg = wx.MessageDialog(self,
                               "This is a small game.\n",
                               "About Tetris",
                               wx.OK | wx.CENTRE | wx.ICON_INFORMATION)
        dlg.ShowModal()
        dlg.Destroy()


    def OnHelp(self, event):
        dlg = wx.MessageDialog(self,
                               "AZERTY keyboard :\n"
                               "--------------------\n"
                               "P \t---------> \tPause\n" 
                               "S \t---------> \tLEFT\n"
                               "F \t---------> \tRIGHT\n"
                               "D \t---------> \tDOWN (rotated right)\n"
                               "E \t----------> \tUP (rotated left)\n"
                               "SPACE \t----> \tAccelerate\n"                               
                               "D \t---------> \tOne line down\n",
                               "Shortcuts",
                               wx.OK | wx.CENTRE | wx.ICON_INFORMATION)
        dlg.ShowModal()
        dlg.Destroy()

#---------------------------------------------------------------------------
        
class Board(wx.Panel):
    """
    ...
    """
    
    BoardWidth = 10
    BoardHeight = 22
    Speed = 300
    ID_TIMER = 1
    
    def __init__(self, *args, **kw):
        super(Board, self).__init__(*args, **kw)

        self.initBoard()

        #------------
        
        # Delete flickers.
        if wx.Platform == "__WXMSW__":
            self.SetDoubleBuffered(True)

    #-----------------------------------------------------------------------
        
    def initBoard(self):
        """
        ...
        """
        
        self.timer = wx.Timer(self, Board.ID_TIMER)
        self.isWaitingAfterLine = False
        self.curPiece = Shape()
        self.nextPiece = Shape()
        self.curX = 0
        self.curY = 0
        self.numLinesRemoved = 0
        self.board = []

        self.isStarted = False
        self.isPaused = False

        self.Bind(wx.EVT_PAINT, self.OnPaint)
        self.Bind(wx.EVT_KEY_DOWN, self.OnKeyDown)
        self.Bind(wx.EVT_TIMER, self.OnTimer, id=Board.ID_TIMER)

        self.clearBoard()


    def shapeAt(self, x, y):
        """
        ...
        """
        
        return self.board[(y * Board.BoardWidth) + x]


    def setShapeAt(self, x, y, shape):
        """
        ...
        """
        
        self.board[(y * Board.BoardWidth) + x] = shape


    def squareWidth(self):
        """
        ...
        """
        
        return self.GetClientSize().GetWidth() // Board.BoardWidth


    def squareHeight(self):
        """
        ...
        """
        
        return self.GetClientSize().GetHeight() // Board.BoardHeight


    def start(self):
        """
        ...
        """
        
        if self.isPaused:
            return

        self.isStarted = True
        self.isWaitingAfterLine = False
        self.numLinesRemoved = 0
        self.clearBoard()

        self.newPiece()
        self.timer.Start(Board.Speed)


    def pause(self):
        """
        ...
        """
        
        if not self.isStarted:
            return

        self.isPaused = not self.isPaused
        statusbar = self.GetParent().statusbar

        if self.isPaused:
            self.timer.Stop()
            statusbar.SetStatusText('paused')
        else:
            self.timer.Start(Board.Speed)
            statusbar.SetStatusText(str(self.numLinesRemoved))

        self.Refresh()


    def clearBoard(self):
        """
        ...
        """
        
        for i in range(Board.BoardHeight * Board.BoardWidth):
            self.board.append(Tetrominoes.NoShape)


    def OnPaint(self, event):
        """
        ...
        """
        
        dc = wx.PaintDC(self)

        size = self.GetClientSize()
        boardTop = size.GetHeight() - Board.BoardHeight * self.squareHeight()

        for i in range(Board.BoardHeight):
            for j in range(Board.BoardWidth):

                shape = self.shapeAt(j, Board.BoardHeight - i - 1)

                if shape != Tetrominoes.NoShape:
                    self.drawSquare(dc,
                        0 + j * self.squareWidth(),
                        boardTop + i * self.squareHeight(), shape)

        if self.curPiece.shape() != Tetrominoes.NoShape:

            for i in range(4):

                x = self.curX + self.curPiece.x(i)
                y = self.curY - self.curPiece.y(i)

                self.drawSquare(dc, 0 + x * self.squareWidth(),
                    boardTop + (Board.BoardHeight - y - 1) * self.squareHeight(),
                    self.curPiece.shape())


    def OnKeyDown(self, event):
        """
        ...
        """
        
        if not self.isStarted or self.curPiece.shape() == Tetrominoes.NoShape:
            event.Skip()
            return

        keycode = event.GetKeyCode()
        print("Keycode :", keycode)

        if keycode == ord('P') or keycode == ord('p'):
            self.pause()
            return

        if self.isPaused:
            return

        #-----------------
        # AZERTY keyboard.
        #-----------------
        # elif keycode == wx.WXK_LEFT:
        elif keycode == ord('S') or keycode == ord('s'):
            self.tryMove(self.curPiece, self.curX - 1, self.curY)

        # elif keycode == wx.WXK_RIGHT:
        elif keycode == ord('F') or keycode == ord('f'):
            self.tryMove(self.curPiece, self.curX + 1, self.curY)

        # elif keycode == wx.WXK_DOWN:
        elif keycode == ord('D') or keycode == ord('d'):
            self.tryMove(self.curPiece.rotatedRight(), self.curX, self.curY)

        # elif keycode == wx.WXK_UP:
        elif keycode == ord('E') or keycode == ord('e'):
            self.tryMove(self.curPiece.rotatedLeft(), self.curX, self.curY)

        elif keycode == wx.WXK_SPACE:
            self.dropDown()

        elif keycode == ord('R') or keycode == ord('r'):
            self.oneLineDown()

        else:
            event.Skip()


    def OnTimer(self, event):
        """
        ...
        """
        
        if event.GetId() == Board.ID_TIMER:

            if self.isWaitingAfterLine:
                self.isWaitingAfterLine = False
                self.newPiece()

            else:
                self.oneLineDown()

        else:
            event.Skip()


    def dropDown(self):
        """
        ...
        """
        
        newY = self.curY

        while newY > 0:
            if not self.tryMove(self.curPiece, self.curX, newY - 1):
                break
            newY -= 1

        self.pieceDropped()


    def oneLineDown(self):
        """
        ...
        """
        
        if not self.tryMove(self.curPiece, self.curX, self.curY - 1):
            self.pieceDropped()


    def pieceDropped(self):
        """
        ...
        """
        
        for i in range(4):

            x = self.curX + self.curPiece.x(i)
            y = self.curY - self.curPiece.y(i)
            self.setShapeAt(x, y, self.curPiece.shape())

        self.removeFullLines()

        if not self.isWaitingAfterLine:
            self.newPiece()


    def removeFullLines(self):
        """
        ...
        """
        
        numFullLines = 0

        statusbar = self.GetParent().statusbar

        rowsToRemove = []

        for i in range(Board.BoardHeight):
            n = 0
            for j in range(Board.BoardWidth):
                if not self.shapeAt(j, i) == Tetrominoes.NoShape:
                    n = n + 1

            if n == 10:
                rowsToRemove.append(i)

        rowsToRemove.reverse()

        for m in rowsToRemove:
            for k in range(m, Board.BoardHeight):
                for l in range(Board.BoardWidth):
                        self.setShapeAt(l, k, self.shapeAt(l, k + 1))

            numFullLines = numFullLines + len(rowsToRemove)

            if numFullLines > 0:

                self.numLinesRemoved = self.numLinesRemoved + numFullLines
                statusbar.SetStatusText(str(self.numLinesRemoved))
                self.isWaitingAfterLine = True
                self.curPiece.setShape(Tetrominoes.NoShape)
                self.Refresh()


    def newPiece(self):
        """
        ...
        """
        
        self.curPiece = self.nextPiece
        statusbar = self.GetParent().statusbar
        self.nextPiece.setRandomShape()

        self.curX = Board.BoardWidth // 2 + 1
        self.curY = Board.BoardHeight - 1 + self.curPiece.minY()

        if not self.tryMove(self.curPiece, self.curX, self.curY):

            self.curPiece.setShape(Tetrominoes.NoShape)
            self.timer.Stop()
            self.isStarted = False
            statusbar.SetStatusText('Game over')


    def tryMove(self, newPiece, newX, newY):
        """
        ...
        """
        
        for i in range(4):

            x = newX + newPiece.x(i)
            y = newY - newPiece.y(i)

            if x < 0 or x >= Board.BoardWidth or y < 0 or y >= Board.BoardHeight:
                return False

            if self.shapeAt(x, y) != Tetrominoes.NoShape:
                return False

        self.curPiece = newPiece
        self.curX = newX
        self.curY = newY
        self.Refresh()

        return True


    def drawSquare(self, dc, x, y, shape):
        """
        ...
        """
        
        colors = ['#000000', '#CC6666', '#66CC66', '#6666CC',
                  '#CCCC66', '#CC66CC', '#66CCCC', '#DAAA00']

        light = ['#000000', '#F89FAB', '#79FC79', '#7979FC',
                 '#FCFC79', '#FC79FC', '#79FCFC', '#FCC600']

        dark = ['#000000', '#803C3B', '#3B803B', '#3B3B80',
                 '#80803B', '#803B80', '#3B8080', '#806200']

        pen = wx.Pen(light[shape])
        pen.SetCap(wx.CAP_PROJECTING)
        dc.SetPen(pen)

        dc.DrawLine(x, y + self.squareHeight() - 1, x, y)
        dc.DrawLine(x, y, x + self.squareWidth() - 1, y)

        darkpen = wx.Pen(dark[shape])
        darkpen.SetCap(wx.CAP_PROJECTING)
        dc.SetPen(darkpen)

        dc.DrawLine(x + 1, y + self.squareHeight() - 1,
            x + self.squareWidth() - 1, y + self.squareHeight() - 1)
        dc.DrawLine(x + self.squareWidth() - 1,
        y + self.squareHeight() - 1, x + self.squareWidth() - 1, y + 1)

        dc.SetPen(wx.TRANSPARENT_PEN)
        dc.SetBrush(wx.Brush(colors[shape]))
        dc.DrawRectangle(x + 1, y + 1, self.squareWidth() - 2,
        self.squareHeight() - 2)

#---------------------------------------------------------------------------
        
class Tetrominoes(object):
    """
    ...
    """
    
    NoShape = 0
    ZShape = 1
    SShape = 2
    LineShape = 3
    TShape = 4
    SquareShape = 5
    LShape = 6
    MirroredLShape = 7

#---------------------------------------------------------------------------
    
class Shape(object):
    """
    ...
    """
    
    coordsTable = (
        ((0, 0),     (0, 0),     (0, 0),     (0, 0)),
        ((0, -1),    (0, 0),     (-1, 0),    (-1, 1)),
        ((0, -1),    (0, 0),     (1, 0),     (1, 1)),
        ((0, -1),    (0, 0),     (0, 1),     (0, 2)),
        ((-1, 0),    (0, 0),     (1, 0),     (0, 1)),
        ((0, 0),     (1, 0),     (0, 1),     (1, 1)),
        ((-1, -1),   (0, -1),    (0, 0),     (0, 1)),
        ((1, -1),    (0, -1),    (0, 0),     (0, 1))
    )
    
    def __init__(self):
        self.coords = [[0,0] for i in range(4)]
        self.pieceShape = Tetrominoes.NoShape

        self.setShape(Tetrominoes.NoShape)

    #-----------------------------------------------------------------------
        
    def shape(self):
        """
        ...
        """
        
        return self.pieceShape


    def setShape(self, shape):
        """
        ...
        """
        
        table = Shape.coordsTable[shape]
        for i in range(4):
            for j in range(2):
                self.coords[i][j] = table[i][j]

        self.pieceShape = shape


    def setRandomShape(self):
        """
        ...
        """
        
        self.setShape(random.randint(1, 7))


    def x(self, index):
        """
        ...
        """
        
        return self.coords[index][0]


    def y(self, index):
        """
        ...
        """
        
        return self.coords[index][1]


    def setX(self, index, x):
        """
        ...
        """
        
        self.coords[index][0] = x


    def setY(self, index, y):
        """
        ...
        """
        
        self.coords[index][1] = y


    def minX(self):
        """
        ...
        """
        
        m = self.coords[0][0]
        for i in range(4):
            m = min(m, self.coords[i][0])

        return m


    def maxX(self):
        """
        ...
        """
        
        m = self.coords[0][0]
        for i in range(4):
            m = max(m, self.coords[i][0])

        return m


    def minY(self):
        """
        ...
        """
        
        m = self.coords[0][1]
        for i in range(4):
            m = min(m, self.coords[i][1])

        return m


    def maxY(self):
        """
        ...
        """
        
        m = self.coords[0][1]

        for i in range(4):
            m = max(m, self.coords[i][1])

        return m


    def rotatedLeft(self):
        """
        ...
        """
        
        if self.pieceShape == Tetrominoes.SquareShape:
            return self

        result = Shape()
        result.pieceShape = self.pieceShape

        for i in range(4):
            result.setX(i, self.y(i))
            result.setY(i, -self.x(i))

        return result


    def rotatedRight(self):
        """
        ...
        """
        
        if self.pieceShape == Tetrominoes.SquareShape:
            return self

        result = Shape()
        result.pieceShape = self.pieceShape

        for i in range(4):
            result.setX(i, -self.y(i))
            result.setY(i, self.x(i))

        return result

#---------------------------------------------------------------------------
    
def main():
    app = wx.App()
    ex = Tetris(None)
    ex.Show(True)
    app.MainLoop()

#---------------------------------------------------------------------------
    
if __name__ == '__main__':
    main()
