
""" 
A conglomeration of all the MSW DLL functions used in the demo program.

Ray Pasco 2010-09-10  
pascor(at)verizon(dot)net

-------------------------------------------------

REQUIREMENTS

1) Any MSW platform that is Windows 2000 or newer.

2) & 3) The PYWIN32 and WMI packages:

    PYWIN32   http://sourceforge.net/projects/pywin32/files/'
    WMI       http://pypi.python.org/pypi/WMI/'

Tested on Win7 (build 7600) 64-bit using Python x86 (32-bit).

Use this at your own risk.
This code is released under the "Official Beerware License" (OBL). If you like 
this program and you happen to run into me someday then offer to by me a beer. 

-------------------------------------------------

NOTES:

1) [ %SYSTEMDRIVE%\AUTOEXEC.BAT ] is called when creating the command shell window
   to ensure that all user account environment variables get defined for use 
   in the command shell.  This is reasonable, but this should be customized 
   for your particular requirements.
   
2) Many calls to time.sleep() are made, though this is not usually acceptable 
   in a wx app. However, this was a "quick and dirty" solution for providing
   delays after window appearance changes. wx.Timer()'s should be used in some fashon 
   to produce reliable delays. I have not found a non-blocking delay function for wx.

-------------------------------------------------

CREDITS:

Mark Hammond's PyWin32 wrapper package for many, many MSW DLL calls.
This allows "pure Python" calls to the DLLs without the explicit use of 
Thomas Heller's ctypes module.

Tim Golden's WinShell Python package. 
A simplified API for calling some MSW DLL's. 

Simon Brunning' FindTopWindow() and FindTopWindows() functions. 
http://www.brunningonline.net/simon/blog/archives/winGuiAuto.py.html

DOCUMENTATION:

Tim Golden has posted easily available usage examples. See:
http://timgolden.me.uk/pywin32-docs/win32api.html
If you wonder how Python on MSW retrieves environment variable values, see:
win32profile.GetEnvironmentStrings @
http://timgolden.me.uk/pywin32-docs/win32profile__GetEnvironmentStrings_meth.html

Mark Hammond's PyWin32 package is a thin 
Python wrapper around the MSW DLLs. He has examples posted, but look for MSW, C, 
C#, VBS, etc., to figure out the number, order and type of argument parameters 
needed to make other Python calls to the MSW DLLs that are not included 
in this module. See: 
http://msdn.microsoft.com/en-us/library/ff468919%28v=VS.85%29.aspx

Some of Simon Brunning's GUI calls (unused in my demo program) show how somme MSW 
OS DLL calls are made using the ctypes module. Any user-created DLL for MSW platforms, 
such as generated by MSW VC, can be called with the help of ctype's data types' constants. 
Borland-generated DLL's must first be converted to the VC DLL type. I understand 
this is a simple one-line command line call to a free Borland utility.

"""

#------------------------------------------------------------------------------

import os, sys
import time     # By not including this crashes the Python interpreter 
                # without giving any traceback message.
import random   # For testing purposes.

import HexCap       as hc

# Make sure the platform is MSW !
# WMI imports PyWin32 which generates its own ImportError if PyWin32 is missing.
if os.name == 'nt' :
    try :
        import wmi
        
    except ImportError :
        print '\n' + sys.argv[ 0 ]
        print
        print '####  MswDllFuncs:'
        print '      Both the PYWIN32 and WMI packages must be installed.\n'
        print '      PYWIN32:  http://sourceforge.net/projects/pywin32/files/'
        print '      WMI:      http://pypi.python.org/pypi/WMI/\n'
        sys.exit(1)
    #end try
else :
    sys.stderr.write( '\n####  MswDllFuncs:    Oops !  This Package can only be run' )
    sys.stderr.write( ' on a MSW platform.\n\n' )
    sys.exit(1)
#end if
    
import array
import ctypes
import ctypes.wintypes
import struct

import win32api
import win32con
import win32com.client
from win32com.shell import shell, shellcon      # for PyWin32 calls
import win32gui
import win32process
import win32ui
import win32event

#------------------------------------------------------------------------------

"""
A "pure Python" wrapper for the GetClassNameA DLL function.

http://packages.python.org/winappdbg/winappdbg.win32.user32-pysrc.html#GetClassNameA

# The C++ prototype:
# int GetClassName( 
#                   HWND hwnd, 687
#                   LPTSTR lpClassName,
#                   int nMaxCount 689  #  );
"""

def _GetClassNameA( hwnd ):
    
    _GetClassNameA          = ctypes.windll.user32.GetClassNameA       # Pywin32 imports windll
    _GetClassNameA.argtypes = [ ctypes.wintypes.HWND, 
                                ctypes.wintypes.LPSTR, 
                                ctypes.wintypes.c_int ]
    _GetClassNameA.restype  = ctypes.c_int
 
    nMaxCount = 0x1000
    dwCharSize = ctypes.wintypes.sizeof( ctypes.c_char_p )
    
    while 1 :
        
        lpClassName = ctypes.create_string_buffer( '', nMaxCount )
        nCount = _GetClassNameA( hwnd, lpClassName, nMaxCount )
        
        if nCount == 0:
            raise ctypes.WinError()
            
        if nCount < nMaxCount - dwCharSize :    # OK: char buffer was large enough.
            break
        
        #-----
        
        nMaxCount += 0x1000         # Try again with a larger buffer.
        
    #end while
        
    return lpClassName.value

#end def

#------------------------------------------------------------------------------

def GetClassNameA( hwnd ) :
    """     """
    #return win32gui.GetClassName( hwnd )   # The "easy" way.
    return _GetClassNameA( hwnd )           # Use the DLL wrapper function above.
    
#end def

#------------------------------------------------------------------------------

def SetCursorPos( coordTuple ) :
    """ Screen-relative, not window-relative. """
    
    if ( type(coordTuple) == type( (1, 2) ) )  or  ( type(coordTuple) == type( [1, 2] ) ) :
        if (len( coordTuple ) >= 2) :
            win32api.SetCursorPos( coordTuple )
    else :
        print
        print '####  MswDllFuncs::SetCursorPos():  Argument Must be a 2-Tuple or 2-Element List'
        print '      Argument Type and Value: ', type( coordTuple ), coordTuple
        print
        sys.exit(1)
    #end if
    
    return
    
#end SetCursorPos def
    
#------------------------------------------------------------------------------

def GetPidFromHwnd( hwnd ) :
    """ Get a pid from a window handle. 
    
    "There can be only one." - Duncan MacLeod from the clan MacLeod.
    """
    try :
        threadId, pid = win32process.GetWindowThreadProcessId( hwnd )
    except :
        pid = None      # The pid is invalid for whatever reason.
    #end try
    
    return pid

#------------------------------------------------------------------------------

def GetConsoleTitle() :
    return win32api.GetConsoleTitle()

#------------------------------------------------------------------------------

def GetScreenSizeY() :
    return win32api.GetSystemMetrics( win32con.SM_CYSCREEN )
    
#------------------------------------------------------------------------------

def GetSelfProcessName() :
    return win32ui.GetName()

#------------------------------------------------------------------------------

def SetFrameTitle( hwnd, title ) :

    try :
        win32gui.SetWindowText( hwnd, title )
    except :
        print '\n####  MswDllFuncs::SetFrameTitle():    Window Is NonExistant : ', 
        if type( hwnd ) == type( 123 ) :
            print hc.HexCap( hwnd )
        else :
            print hwnd
        #end if
    #end try
    
#end def

#------------------------------------------------------------------------------

def SetForegroundWindow( hwnd ) :
    """ Makes this frame topmost of all visible process windows. """
    
    try :
        win32gui.SetForegroundWindow( hwnd )
    except :
        pass
    #end try
    
#end def

#------------------------------------------------------------------------------

def DisableWinInput( hwnd ) :
    """ Disables the mouse and keyboard. """
    
    try :
        win32gui.EnableWindow( hwnd, False )
    except :
        pass    # Don't even bother informing about an error.
    #end try
    
#end def

#------------------------------------------------------------------------------

def EnableWinInput( hwnd ) :
    """ Re-enables the mouse and keyboard. """
    
    try :
        win32gui.EnableWindow( hwnd, True )
    except :
        pass    # Don't even bother informing about an error.
    #end try
    
#end def

#------------------------------------------------------------------------------

def HideWin( hwnd ) :
    """Make the window invisible.  *** Makes the Task Bar button invisible, too. *** """
    
    try :
        win32gui.ShowWindow( hwnd, win32con.SW_HIDE )    # ? More than 2 available mode constants ?
    except :
        pass    # Don't even bother informing about an error.
    #end try
    
#end def

#------------------------------------------------------------------------------

def ShowWin( hwnd ) :
    """Make the window visible as well as its Task Bar button.  """
    
    try :
        win32gui.ShowWindow( hwnd, win32con.SW_NORMAL )    # ? More than 2 available mode constants ?
    except :
        pass    # Don't even bother informing about an error.
    #end try
    
#end def

#------------------------------------------------------------------------------

def BlinkWindowHwnd( hwnd, duration=None, anteDelay=None, postDelay=None ) :
    """ A MSW DLL process creation function call customized to create a command shell window.
    """
    
    offTime = 1.00     # the default off duration
    if duration != None :    
        offTime = duration
    
    if anteDelay != None :
        time.sleep( anteDelay )
    
    try :
        HideWin( hwnd )
    except :
        print '\n####  MswDllFuncs::BlinkWindowHwnd():    Window Is NonExistant : ', hc.HexCap( hwnd )
        return
    #end try
    
    time.sleep( offTime )
    try :
        ShowWin( hwnd )
    except :
        print '\n####  MswDllFuncs::BlinkWindowHwnd():    Window Is NonExistant : ', hc.HexCap( hwnd )
        return
    #end try
    
    if postDelay != None :
        time.sleep( postDelay )
    
#end def

#------------------------------------------------------------------------------

def CreateExecProc( execCmdline ) :
    """ Run an executable program with any command line arguments.
    Returns the PID and execution success status, unlike os.system().

    The [ process.Create() ] call seems to be able to kick off any type of executable, 
    but much more investigation is needed.

    The PID may be used in other PyWin32 calls to manipulate the process window, 
    e.g., getting/changing its position, blinking it and hiding/reshowing it, 
    or to simply kill the process.

    """
    
    wmiObject = wmi.WMI()           # Part of WinShell
    win32ProcessCreateMethod = wmiObject.Win32_Process
    
    pid, success = win32ProcessCreateMethod.Create( CommandLine=execCmdline )
    
    return pid, success      
    
#end CreateExecProc

#--------------------------------------------------------------------

def CreateCmdShell() :
    """ Create an MSW command shell window process. 
    Returns the PID and execution success status, unlike os.system().
    """
    
    """ 
    Manually inserting ampersands chars into a string is a PITA.
    Note the surrounding space chars. These are not necessary, but will format
    the composite command string to a more human-readable form if you print out 
    the command line string.
    """
    DAS = '&&  '        # Double AmperSands
    
    #--------------
    
    def CmdJoin( cmd_str, cmdAppend_str ) :
        """ A much needed utility function. """
        
        return cmd_str + DAS + cmdAppend_str    # Every def should be this simple !
    
    #--------------
    
    procCreationObj = wmi.WMI()               # A MSW process creation object (?)
    process = procCreationObj.Win32_Process   # An alias
    
    """
    The following is a convenient way to generate the proper syntax for a 
    MSW composite command string. Note that MSW commands separated by a DAS [ && ] 
    execute the latter command whether or not the previous command successfully executed. 
    This is opposed to using a single ampersand [ & ] which terminates execution of 
    the rest of the composite command string if the immediately previous command fails.
    """
    # Separate MSW commands to be strung together into a single composite 
    #   command line string.
    # Each will be separated by a DAS [ && ] in the final command string.
    #
    cmds_list = [ 
        
        'CMD.exe /K CLS.exe',   # Create a command shell window. 
                                # [ /K ] keeps it open after all commands have completed.
                                
        'IF EXIST %SYSTEMDRIVE%\\AUTOEXEC.BAT CALL %SYSTEMDRIVE%\\AUTOEXEC.BAT', 
                                # Any environment definitions in here will be in effect.
                                # All "System Properties" env vars are always in effect.
        
        'SET PROMPT=$P $G ',    # Set a useful command prompt string. Note the trailing space.
                                #  This string is "space-chars-sensitive" to MSW.
                                # CmdJoin() appends one space char after [ $G ] that is 
                                # needed to separate the prompt char [ > ] from the cursor.
                                
        'CLS',                  # Clear the window of the previous command execution messages.
        
        'ECHO.',                # Write a blank line to the window.
        
        'CD',                   # Display the CWD
        
        ]
    
    # Concatenate the indivual complete commands into a single looong command string.
    # Vista and Win7 allow extremely long command strings.
    for idx in xrange( len(cmds_list) ) :
    
        if (idx == 0) :
            cmdLine = cmds_list[ 0 ]    # initialize the command string.
        else :
            cmdLine = CmdJoin( cmdLine, cmds_list[ idx ] )
        #end if
        
    #end for
    #print '\n----  cmdLine  [ %s ]' % (cmdLine)    # Uncomment to view the composite command.
    
    # The main point of this function:
    pid, success = procCreationObj.Win32_Process.Create( CommandLine=cmdLine )
    
    # This call can be used to create ANY independent process:
    # However, the default folder is set to a system default folder.
    # PID, success = process.Create( CommandLine='Notepad.exe ZZZ.TXT' )    # A WinShell call.
    
    return pid, success      
    
#end CreateCmdShell

#------------------------------------------------------------------------------

def FindTopWindows( wantedText=None, wantedClass=None, selectionFunction=None ) :
    """ Return a hwnd ID list of all the top level process frame windows using 
    three available search options.
    
    You can identify windows using titles and/or classes. 
    Or, instead use a selection function.

    Arguments
    ---------
    
    wantedText          Text which required window's frame title must contain.
    
    wantedClass         Class to which required window must belong.
    
    selectionFunction   Window selection function. Reference to a function
                        should be passed here. The function should take hwnd as
                        an argument, and should return True when passed the
                        hwnd of a desired window.

    Returns :           A list containing the window handles (hwnd's) of all 
                        top level windows matching the supplied selection criteria.

    Usage example :     optDlgHwnd_list = FindTopWindows( wantedText="Options" )
    
    http://www.brunningonline.net/simon/blog/archives/winGuiAuto.py.html

    # Programmer  : Simon Brunning - simon(at)brunningonline(dot)net
    # Date        : 25 June 2003
    # Version     : 1.0 pre-alpha 2
    # Copyright   : Released to the public domain. Provided as-is, with no warranty.
    # Notes       : Requires Python 2.3, win32all and ctypes 

    """
    topWindows = []     # will be populated
    win32gui.EnumWindows( _WindowEnumerationHandler, topWindows )
    
    results = []
    for hwnd, windowText, windowClass in topWindows :
    
        if wantedText and not _NormalizeText( wantedText ) in _NormalizeText( windowText ) :
            continue
            
        if wantedClass and not windowClass == wantedClass :
            continue
            
        if selectionFunction and not selectionFunction( hwnd ) :
            continue
        
        results.append( hwnd )      
        
    #end for
    
    return results      # list of hwnd's
    
#end FindTopWindows def
    
#------------------------------------------------------------------------------

def _WindowEnumerationHandler( hwnd, resultList ) :
    """ Utility function to pass to win32gui.EnumWindows() to generate list 
    of window handles, window text titles, and window class type tuples.
    """
    resultList.append( ( hwnd,
                         win32gui.GetWindowText( hwnd ),
                         win32gui.GetClassName(  hwnd ) ) )
#end def

#------------------------------------------------------------------------------

def _NormalizeText( controlText ) :
    """ Utility function to remove [ & ] characters and make lower case.
    
    Useful for matching control text.
    """
    
    return controlText.lower().replace( '&', '' )
    
#end def

#------------------------------------------------------------------------------

def FindTopWindow( wantedText=None, wantedClass=None, selectionFunction=None, 
                   warnMultiple=True, returnMultiple=False ) :
    """ Return a hwnd ID of a top level process frame windows using 
    three available search options.
    
    You can identify windows using titles and/or classes. 
    Or, instead use a selection function.

    Arguments :
    
        wantedText          Text which required window's frame title must contain.

        wantedClass         Class to which required window must belong.

        selectionFunction   Window selection function. Reference to a function
                            should be passed here. The function should take hwnd as
                            an argument, and should return True when passed the
                            hwnd of a desired window.

    Returns :           A list containing the window handles of all top level
                        windows matching the supplied selection criteria.
                    
    Raises :
    
        WinGuiAutoError     When no window found.

    Usage example :      optDialog = FindTopWindow( wantedText="Options" )
    
    """
    
    # There can be more than 1 app frame (top window) that meets the given requirements.
    topWindows_list = FindTopWindows( wantedText, wantedClass, selectionFunction )
    
    if topWindows_list :
        
        numFoundWindows = len( topWindows_list )
        if (numFoundWindows > 1)  and  warnMultiple :
            print '\n!!!!  FindTopWindow():    Warning:   Multiple Top Windows Found: ', numFoundWindows
        
        if returnMultiple :
            topWindow = topWindows_list       # return all found
        else:
            topWindow = topWindows_list[0]    # return only the first matching window
        #end if
        
    else :
    
        print '\n####  MswDllFuncs::FindTopWindow():    No Top Level Window Found with :'
        if wantedText :
            print '\t wantedText        = [', wantedText, ']'
            
        if wantedClass :
            print '\t wantedClass       = ',  repr( wantedClass )
        
        if selectedFunction :
            print '\t selectionFunction = ',  repr( selectionFunction )
        
        topWindow = None
        
    #end if
    
    return topWindow

#end FindTopWindow def

#------------------------------------------------------------------------------

def KillProcHwnd( hwnd ) :
    """ Kill the process associated with the givenwindow handle.
    Silently ignores a nonexistant handle.
        
    http ://timgolden.me.uk/python/win32_how_do_i/find-the-window-for-my-subprocess.html
    
    The challenge : to find the windows belonging to the process you've just kicked off.
    
    The idea is that if, for example, you run a notepad session, you then want to read 
    the text entered into it, or close it down if it's been open too long, or whatever. 
    The problem is that Windows doesn't provide a straightforward mapping from the PID
    (process ID) to a window handle (hwnd). 
    
    The situation is complicated because some process may not have a window, or may have 
    several. The approach below uses the venerable technique of iterating over all 
    top-level windows and finding the ones belonging to a PID. It only considers windows 
    which are visible and enabled (which is generally what you want) and returns a list 
    of the ones associated with your PID.
    
    """
    win32gui.SendMessage( hwnd, win32con.WM_CLOSE, 0, 0 )
    
#end def

#------------------------------------------------------------------------------

def GetHwndFromPid( pid ) :

    def CallBackFunc( hwnd, hwnds ) :
        
        if win32gui.IsWindowVisible( hwnd ) and win32gui.IsWindowEnabled( hwnd ) :
            
            threadId, pidFound = win32process.GetWindowThreadProcessId( hwnd )
            
            if pidFound == pid :    
                hwnds.append( hwnd )
        else :
            return None
        #end if
        
        return True
        
    #end CallBackFunc def
    
    #-----------------------------
    
    # Retrieve all the visible top level windows.
    # ??? How can INVISIBLE top windows be retrieved ???
    hwnds_list = []
    win32gui.EnumWindows( CallBackFunc, hwnds_list )
    #print '\n>>>>  GetHwndFromPid():   win32gui.EnumWindows() hwnds_list', hwnds_list
    
    hwnd = None     # Flag value:  the pid has not been found (yet).
    if hwnds_list :
    
        hwnd = hwnds_list[0]        # There can be only one, at most, using the pid.
        #print '\n----  GetHwndFromPid():    hwnd ', hc.HexCap( hwnd )
        
    else :
        print '\n!!!!  NOTE:   GetHwndFromPid():   ',
        print 'Hwnd NOT FOUND For PID', pid, 
        if pid != None :    
            print hc.HexCap( pid )
        else :
            print
        #end if
    #end if
    
    return hwnd
    
#end GetHwndFromPid def

#------------------------------------------------------------------------------

def KillProcPid( pid ) :
    """ Various ways to kill a process indirectly using its PID.
    There is no direct way to "politely" close an app without its hwnd.
    A process may be killed by using either its hwnd or its process handle.
    """
    
    method = random.randint( 1, 2 )     # 3 is a "brute force" kill method.
    if method == 1 :
        """
        This ignores a nonexistant PID.
        This approach is indirect since it uses the hwnd, not the pid.
        The "main" process window must be queried and and used to do this.
        MSDN recommends this way since it queries the process to see
        if is hung and to wait for the user to save his work before issuing
        the terminate command.
        """
        hwnd = GetHwndFromPid( pid )
        if hwnd != None :
            # If a process such as Notepad has unsaved changes the following call
            #   will popup a message to the user. The call will wait until the
            #   user has replied.  [ Cancel ] or [ Save > Cancel ] will cause the call 
            #   to return [ False ]. So, the call will wait until [ Cancel ]
            #   or [ Save > {filename} ] is entered by the user.
            #
            win32gui.SendMessage( hwnd, win32con.WM_CLOSE )  # Args 3 & 4 not used.
            
        else :
            # The hwnd is invalid for whatever reason.
            pass
        #end if
    
    elif method == 2 :    # A "nice" termination procedure. Requires hwnd.
        """
        This sequence effectively does exactly the same thing as the call:
            win32gui.SendMessage( hwnd, win32con.WM_CLOSE, 0, 0 )
            
        Algorithm from http://msdn.microsoft.com/en-us/magazine/cc301501.aspx
        "MSDN Magazine", August 2002. A semi-coherent article.
        """
        
        flags = win32con.PROCESS_TERMINATE | win32con.SYNCHRONIZE
        procHandle = win32api.OpenProcess( flags, 0, pid )  # get the process handle
        
        # There is no direct way to "politely" close an app without its hwnd.
        hwnd = GetHwndFromPid( pid )
        if not hwnd :       # "He's dead, Jim." - Dr. Leonard McCoy, AKA "Bones".
            # The hwnd is invalid for whatever reason.
            return
        #end if
        
        # If a process such as Notepad has unsaved changes then the following call
        #   will cause the app to popup a message to the user. The call will wait 
        #   until the app is satisfied that the user has replied.  [ Cancel ] or 
        #   [ Save > Cancel ] will cause the call to return [ False ]. 
        #   We will loop until [ True ] is returned.
        
        flags = win32con.SMTO_ABORTIFHUNG | win32con.SMTO_NOTIMEOUTIFNOTHUNG
        readyToDie = False      # Loop entry condition.
        while not readyToDie :
            
            rc, readyToDie = win32gui.SendMessageTimeout(  \
                                hwnd, win32con.WM_QUERYENDSESSION, 0, 0, flags, 1500 )
        #end loop
        
        # readyToDie == True, gauranteed.
        try :
            win32api.TerminateProcess( procHandle, 0 )
        except :
            pass
        #end try
        
    elif method == 3 :    # This kills a process "with prejudice".
        
        flags = win32con.PROCESS_TERMINATE | win32con.SYNCHRONIZE
        procHandle = win32api.OpenProcess( flags, 0, pid ) # get process handle
        win32api.TerminateProcess( procHandle, -1)    # kill by process handle
        win32api.CloseHandle( procHandle )            # the close api
        
    #end if
    
#end def

#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

"""
Retrieves the MSW frame title text.
"""

def GetWinTitle( hwnd ) :
    
    return win32gui.GetWindowText( hwnd )
    
#end def

#------------------------------------------------------------------------------

def GetWinRect( hwnd ) :
    """ MS's window size and position query function. The compliment to this 
    function is win32gui.MoveWindow(). In MS's infinite wisdom, GetWinRect()
    returns the U-L and L-R coordinates of the window. However, MoveWindow
    requires the new U-R coordinate and the new *SIZE* measurements rather than 
    a new L-R coordinate. This is simply brilliant thinking.
    
    """
    winRect = None      # First set return to the default error flag value.
    try :
        winRect = win32gui.GetWindowRect( hwnd )
    except :
        #print
        #print '####  MswDllFuncs::GetWinRect():    Window No Longer Exists : ',
        #if type( hwnd ) == type( 123 ) :    print hc.HexCap( hwnd )
        #else :                              print hwnd
        
        return None     
    #end try
    
    return winRect
    
#end def

#------------------------------------------------------------------------------

def GetWinPosition( hwnd ) :
    
    winRect = GetWinRect( hwnd )
    
    posnX  = winRect[0]
    posnY  = winRect[1]
    
    return (posnX, posnY)
    
#end def

#------------------------------------------------------------------------------

def SetWinPosition( hwnd, posnX, posnY ) :
    """ There is no MSW equivalent function [ GetWindowPos() ].
    Use this module's [ GetWindowPosition() ]. """
    
    win32gui.SetWindowPos( hwnd, posnX, posnY )
    
#end def

#------------------------------------------------------------------------------

def GetWinSize( hwnd ) :
    
    winRect = GetWinRect( hwnd )
    if winRect == None :    return None     # Window no longer exists.
    
    # Calculate the window's size
    startX  = winRect[0]
    startY  = winRect[1]
    rightX  = winRect[2]
    bottomY = winRect[3]
    
    sizeX = rightX  - startX
    sizeY = bottomY - startY
    
    return (sizeX, sizeY)
    
#end def

#------------------------------------------------------------------------------

def SetWinSize( hwnd, sizeX, sizeY ) :
    """ Resize a window without moving it. """
    
    # Find the window's current position so it stays this way.
    posnOrigX, posnOrigY = GetWinPosition( hwnd )
    
    # Calculate the new ending ordinates.
    newRightX  = posnOrigX + sizeX
    newBottomY = posnOrigY + sizeY
    
    # Resize the window keeping the original position.
    bRepaint = True        # Repaint as necessary
    win32gui.MoveWindow( hwnd, posnOrigX, posnOrigY, sizeX, sizeY, bRepaint )
    
#end def

#------------------------------------------------------------------------------

def MoveWin( hwnd, destX, destY ) :
    """ Move a window without resizing it. """
    
    # Find the window's current size so it stays this way.
    sizeX, sizeY = GetWinSize( hwnd )
    posnX, posnY = GetWinPosition( hwnd )
    
    # Move the window ito the destination position.
    bRepaint = True        # Repaint as necessary
    SetWinPosition( hwnd, destX, destY )
                         
#end def

#------------------------------------------------------------------------------

def GrowWinSize( hwnd, targetX, targetY ) :
    
    startX, startY = GetWinSize( hwnd )
    incrX = 1
    deltaX = targetX - startX
    if deltaX < 0 :    incrX = 0 - incrX
    
    deltaY = targetY - startY
    incrY = 1
    deltaY = targetY - startY
    if deltaY < 0 :    incrY = 0 - incrY
    
    iters = max( abs( deltaX ), abs( deltaY ) )
    nextX = startX
    nextY = startY
    for iter in xrange( iters ) :
        
        actualX, actualY = GetWinSize( hwnd )
        
        if ((deltaX > 0)  and (targetX > actualX))  or  \
           ((deltaX < 0)  and (targetX < actualX))  :
            nextX += incrX
        
        if ((deltaY > 0)  and (targetY > actualY))  or  \
           ((deltaY < 0)  and (targetY < actualY))  :
            nextY += incrY
        
        SetWinSize( hwnd, nextX, nextY )
        
        time.sleep( 0.010 )
        
    #end while
    
    actualX, actualY = GetWinSize( hwnd )
    
#end def

#------------------------------------------------------------------------------

def ShrinkWindow( hwnd ) :
    """ Reduce a window size from its current size to (0, 0) without moving it. """
    
    sizeOrigX, sizeOrigY = GetWinSize( hwnd )
            
    # Shring the window to size (0, 0) in [ numSteps ].
    numSteps = 50
    bRepaint = True        # Repaint as necessary
    for stepNum in xrange( numSteps ) :
        
        sizeStepX = int( sizeOrigX * (float(numSteps - stepNum) / numSteps) )
        sizeStepY = int( sizeOrigY * (float(numSteps - stepNum) / numSteps) )
        
        SetWinSize( hwnd, sizeStepX, sizeStepY )
         
    #end for
    
#end ShrinkWindow def

#------------------------------------------------------------------------------

def SlideWindowFx( hwnd, destPosnX, destPosnY, fx=True ) :
    """ Relocate a window using "animation", that is, pseudo-smooth motion 
    in 50 steps. Its original size is retored.
    """
    
    # Get the wind's starting size.
    try :
        sizeOrigX, sizeOrigY = GetWinSize( hwnd )
        
    except TypeError :
        print '\n####  MswDllFuncs::SlideWindowFx():    Given Window Does Not Exist : ',
        if type(hwnd) == type(123) :
            print hc.HexCap( hwnd )
        else :
            print hwnd
        #end if
        
        return
    #end if
    
    # The position may have to be reset if either ordinate is negative.
    # The DLL call makes no move at all if it gets negative ordinate values.
    endPosnX, endPosnY = destPosnX, destPosnY   # Where to reposition.
    
    if (endPosnX < 0) :    endPosnX = 0     # Negative value error position
    if (endPosnY < 0) :    endPosnY = 0
    
    screenX = win32api.GetSystemMetrics(0)  # The first of many elements.
    screenY = win32api.GetSystemMetrics(1)
    
    sizeOrigX, sizeOrigY = GetWinSize( hwnd )
    posnOrigX, posnOrigY = GetWinPosition( hwnd )
    
    # Find the total distance to move
    deltaX = endPosnX - posnOrigX
    deltaY = endPosnY - posnOrigY    # - winVertSizeFudgeFactor
    
    # Move the window in [ numSteps ] to the final position.
    numSteps = 50
    bRepaint = True        # Repaint as necessary
    for stepNum in xrange( 1, numSteps+1 ) :
        
        time.sleep( 0.010 )
        stepPosnX = posnOrigX + ( deltaX * stepNum / numSteps )
        stepPosnY = posnOrigY + ( deltaY * stepNum / numSteps )
        
        try :
            win32gui.MoveWindow( hwnd, stepPosnX, stepPosnY, 
                                 sizeOrigX, sizeOrigY, bRepaint )
        except :
            print
            print '####  MswDllFuncs::SlideWindowFx():    Window No Longer Exists.'
            print '      hwnd : ', hc.HexCap( hwnd )
            return
        #end try
    #end for
    
    # Correct for any final-size error. 
    # Move to the possibly-negative-given-value-error position, 
    #   not necessarily the given position.
    try :
        win32gui.MoveWindow( hwnd, endPosnX, endPosnY, 
                             sizeOrigX, sizeOrigY, bRepaint )
    except :
        print
        print '####  MswDllFuncs::SlideWindowFx():    Window No Longer Exists.'
        print '      hwnd : ', hc.HexCap( hwnd )
        return
    #end try
    
    #winSizeSlideX, winSizeSlideY = GetWinSize( hwnd )
    #print
    #print '####  MswDllFuncs::SlideWindowFx():  winSizeSlideX, winSizeSlideY ',  \
    #                               winSizeSlideX, winSizeSlideY
    
#end SlideWindowFx def

#------------------------------------------------------------------------------

""" 
Converts each path, filename or path\filename element to its 8-char 
or 8.3 representation.

This is handy if you want a shorter path string or a string that 
contains no embedded spaces and non-alphanumeric symbols
other than tilde [ ~ ].

Ray Pasco
2010-08-27
pascor(at)verizon(dot)net

I release this code under the "Official Beerware License".
"""

def GetShortPathname( pathname ) :
    """ Converts each path, filename or path+filename element to its 8-char 
    or 8.3 representation.
    
    This is handy if you want a shorter path string or a string that contains 
    no embedded spaces or non-alphanumeric symbols other than tilde [ ~ ].
    """
    
    return win32api.GetShortPathName( pathname )
    
#end def

#------------------------------------------------------------------------------

"""
Search for a filename [ basename.ext ] using the MSW environment varaiable %PATH%

Ray Pasco
2010-09-03
pascor(at)verizon(dot)net

TO DO:

1. Implement the "recurse" subfolders switch. This is sorely needed.

"""

def SearchFileInEnvPath( filename, shortName=False, allPaths=False, recurse=False ) :
    """ Find the first folder in which [ filename ] exists.
    
    Set [ shortName ] to True to return a short pathname.
    
    Set [ allPaths ] to True to return a list of all paths
    in which filename is found - even if <= 1 path is found.
    
    The entries in the %PATH% env var are inherently full paths. 
    """
    
    if allPaths :
        path = []      # return a list no matter how many files are found
    else :
        path = None
    #end if
    
    pathList = os.environ.get( 'path' ).split( ';' )
    
    #for aPath in pathList :
    #    if aPath :
    #        print '[ %s ]' % (aPath)
    #print
    
    for searchPath in pathList :
        
        fullPathname = os.path.join( searchPath.strip(), filename.strip() )
        #print '\n----  searchPath, fullPathname ', searchPath, fullPathname
        
        if os.path.exists( fullPathname ) :
            
            foundPath, _ = os.path.split( fullPathname )
            if shortName :
                # gsp.GetShortPathname( foundPath )     # alternative call
                foundPath = win32api.GetShortPathName( foundPath )
            #end if
            
            if allPaths :
                path.append( foundPath )
            else :
                break       # Search no more even if others exist.
            #end if
            
        #end if
        
    #end for
    
    return path
    
#end SearchFileInEnvPath def

#------------------------------------------------------------------------------

"""
http ://www.brunningonline.net/simon/blog/archives/winGuiAuto.py.html

# Module      : winGuiAuto.py
# Synopsis    : Windows GUI automation utilities
# Programmer  : Simon Brunning - simon@brunningonline.net
# Date        : 25 June 2003
# Version     : 1.0 pre-alpha 2
# Copyright   : Released to the public domain. Provided as-is, with no warranty.
# Notes       : Requires Python 2.3, win32all and ctypes 

http://www.brunningonline.net/simon/blog/archives/winGuiAuto.py.html

Windows GUI automation utilities.

Until I get around to writing some docs and examples, the tests at the foot of
this module should serve to get you started.
"""

#-----------------------

def DumpWindowControls( hwnd ) :
    """Dump all controls from a window into a nested list
    Useful during development, allowing to you discover the structure of the
    contents of a window, showing the text and class of all contained controls.

    Arguments :      The window handle of the top level window to dump.

    Returns         A nested list of controls. Each entry consists of the
                    control's hwnd, its text, its class, and its sub-controls,
                    if any.

    Usage example :  replaceDialog = FindTopWindowMsw( wantedText='Replace' )
                    pprint.pprint( DumpWindowControls( replaceDialog ) )
    """
    windows = []
    try :
        win32gui.EnumChildWindows( hwnd, _WindowEnumerationHandler, windows )
    except win32gui.error :
        # No child windows
        return
    windows = [list( window ) for window in windows]
    for window in windows :
        childHwnd, windowText, windowClass = window
        window_content = DumpWindowControls( childHwnd )
        if window_content :
            window.append( window_content )
    return windows

#------------------------------------------------------------------------------

def FindControl( topHwnd,
                 wantedText=None,
                 wantedClass=None,
                 selectionFunction=None ) :
    """Find a control.
    You can identify a control using caption, classe, a custom selection
    function, or any combination of these. ( Multiple selection criteria are
    ANDed. If this isn't what's wanted, use a selection function. )

    Arguments :
    topHwnd             The window handle of the top level window in which the
                        required controls reside.
    wantedText          Text which the required control's captions must contain.
    wantedClass         Class to which the required control must belong.
    selectionFunction   Control selection function. Reference to a function
                        should be passed here. The function should take hwnd as
                        an argument, and should return True when passed the
                        hwnd of the desired control.

    Returns :            The window handle of the first control matching the
                        supplied selection criteria.
                    
    Raises :
    WinGuiAutoError     When no control found.

    Usage example :      optDialog = FindTopWindowMsw( wantedText="Options" )
                        okButton = FindControl( optDialog,
                                               wantedClass="Button",
                                               wantedText="OK" )
                        """
    controls = FindControls( topHwnd,
                            wantedText=wantedText,
                            wantedClass=wantedClass,
                            selectionFunction=selectionFunction )
    if controls :
        return controls[0]
    else :
        raise WinGuiAutoError( "No control found for topHwnd: " +
                               repr( topHwnd ) +
                               ", wantedText=" +
                               repr( wantedText ) +
                               ", wantedClass=" +
                               repr( wantedClass ) +
                               ", selectionFunction=" +
                               repr( selectionFunction ) )

#------------------------------------------------------------------------------

def FindControls( topHwnd, wantedText=None, wantedClass=None, selectionFunction=None ) :
    """Find controls within a MSW window.
    You can identify controls using captions, classes, a custom selection
    function, or any combination of these. ( Multiple selection criteria are
    ANDed. If this isn't what's wanted, use a selection function. )

    Arguments :
    topHwnd             The window handle of the top level window in which the
                        required controls reside.
    wantedText          Text which the required controls' captions must contain.
    wantedClass         Class to which the required controls must belong.
    selectionFunction   Control selection function. Reference to a function
                        should be passed here. The function should take hwnd as
                        an argument, and should return True when passed the
                        hwnd of a desired control.

    Returns :            The window handles of the controls matching the
                        supplied selection criteria.    

    Usage example :      optDialog = FindTopWindowMsw( wantedText="Options" )
                        def findButtons( hwnd, windowText, windowClass ) :
                            return windowClass == "Button"
                        buttons = FindControl( optDialog, wantedText="Button" )
                        """
    def searchChildWindows( currentHwnd ) :
        results = []
        childWindows = []
        try :
            win32gui.EnumChildWindows( currentHwnd,
                                      _WindowEnumerationHandler,
                                      childWindows )
        except win32gui.error :
            # This seems to mean that the control *cannot* have child windows,
            # i.e. not a container.
            return
        for childHwnd, windowText, windowClass in childWindows :
            descendentMatchingHwnds = searchChildWindows( childHwnd )
            if descendentMatchingHwnds :
                results += descendentMatchingHwnds

            if wantedText and \
               not _NormaliseText( wantedText ) in _NormaliseText( windowText ) :
                continue
            if wantedClass and \
               not windowClass == wantedClass :
                continue
            if selectionFunction and \
               not selectionFunction( childHwnd ) :
                continue
            results.append( childHwnd )
        return results

    return searchChildWindows( topHwnd )

#------------------------------------------------------------------------------

def GetTopMenu( hwnd ) :
    """Get a window's main, top level menu.
    
    Arguments :
    hwnd            The window handle of the top level window for which the top
                    level menu is required.

    Returns :        The menu handle of the window's main, top level menu.

    Usage example :  hMenu = GetTopMenu( hwnd )"""
    return ctypes.windll.user32.GetMenu( ctypes.c_long( hwnd ) )

#------------------------------------------------------------------------------

def ActivateMenuItem( hwnd, menuItemPath ) :
    """Activate a menu item
    
    Arguments :
    hwnd                The window handle of the top level window whose menu you 
                        wish to activate.
    menuItemPath        The path to the required menu item. This should be a
                        sequence specifying the path through the menu to the
                        required item. Each item in this path can be specified
                        either as an index, or as a menu name.
                    
    Raises :
    WinGuiAutoError     When the requested menu option isn't found.

    Usage example :      ActivateMenuItem( notepadWindow, ( 'file', 'open' ) )
    
                        Which is exactly equivalent to...
                    
                        ActivateMenuItem( notepadWindow, ( 0, 1 ) )"""
    # By Axel Kowald ( kowald@molgen.mpg.de )
    # Modified by S Brunning to accept strings in addition to indicies.

    # Top level menu    
    hMenu = GetTopMenu( hwnd )
    
    # Get top level menu's item count. Is there a better way to do this?
    for hMenuItemCount in xrange( 256 ) :
        try :
            GetMenuInfo( hMenu, hMenuItemCount )
        except WinGuiAutoError :
            break
    hMenuItemCount -= 1
    
    # Walk down submenus
    for submenu in menuItemPath[ :-1] :
        try : # submenu is an index
            0 + submenu
            submenuInfo = GetMenuInfo( hMenu, submenu )
            hMenu, hMenuItemCount = submenuInfo.submenu, submenuInfo.itemCount
        except TypeError : # Hopefully, submenu is a menu name
            try :
                dump, hMenu, hMenuItemCount = _FindNamedSubmenu( hMenu,
                                                                hMenuItemCount,
                                                                submenu )
            except WinGuiAutoError :
                raise WinGuiAutoError( "Menu path " +
                                      repr( menuItemPath ) +
                                      " cannot be found." )
           
    # Get required menu item's ID. ( the one at the end ).
    menuItem = menuItemPath[-1]
    try : # menuItem is an index
        0 + menuItem
        menuItemID = ctypes.windll.user32.GetMenuItemID( hMenu,
                                                        menuItem )
    except TypeError : # Hopefully, menuItem is a menu name
        try :
            subMenuIndex, dump, dump = _FindNamedSubmenu( hMenu,
                                        hMenuItemCount,
                                        menuItem )
        except WinGuiAutoError :
            raise WinGuiAutoError( "Menu path " +
                                  repr( menuItemPath ) +
                                  " cannot be found." )
        # TODO - catch WinGuiAutoError. and pass on with better info.
        menuItemID = ctypes.windll.user32.GetMenuItemID( hMenu, subMenuIndex )

    # Activate    
    win32gui.PostMessage( hwnd, win32con.WM_COMMAND, menuItemID, 0 )
    
#------------------------------------------------------------------------------

def GetMenuInfo( hMenu, uIDItem ) :
    """Get various info about a menu item.
    
    Arguments :
    hMenu               The menu in which the item is to be found.
    uIDItem             The item's index

    Returns :            Menu item information object. This object is basically
                        a 'bunch'
                        ( see http ://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/52308 ).
                        It will have useful attributes : name, itemCount,
                        submenu, isChecked, isDisabled, isGreyed, and
                        isSeperator
                    
    Raises :
    WinGuiAutoError     When the requested menu option isn't found.       

    Usage example :      submenuInfo = GetMenuInfo( hMenu, submenu )
                        hMenu, hMenuItemCount = submenuInfo.submenu, submenuInfo.itemCount"""
    # An object to hold the menu info
    class MenuInfo( Bunch ) :
        pass
    menuInfo = MenuInfo()

    # Menu state    
    menuState = ctypes.windll.user32.GetMenuState( hMenu,
                                                  uIDItem,
                                                  win32con.MF_BYPOSITION )
    if menuState == -1 :
        raise WinGuiAutoError( "No such menu item, hMenu=" +
                               str( hMenu ) +
                               " uIDItem=" +
                               str( uIDItem ) )
    menuInfo.isChecked = bool( menuState & win32con.MF_CHECKED )
    menuInfo.isDisabled = bool( menuState & win32con.MF_DISABLED )
    menuInfo.isGreyed = bool( menuState & win32con.MF_GRAYED )
    menuInfo.isSeperator = bool( menuState & win32con.MF_SEPARATOR )
    # ... there are more, but these are the ones I'm interested in
    
    # Menu name
    menuName = ctypes.c_buffer( "\000" * 32 )
    ctypes.windll.user32.GetMenuStringA( ctypes.c_int( hMenu ),
                                        ctypes.c_int( uIDItem ),
                                        menuName, ctypes.c_int( len( menuName ) ),
                                        win32con.MF_BYPOSITION )
    menuInfo.name = menuName.value

    # Sub menu info
    menuInfo.itemCount = menuState >> 8
    if bool( menuState & win32con.MF_POPUP ) :
        menuInfo.submenu = ctypes.windll.user32.GetSubMenu( hMenu, uIDItem )
    else :
        menuInfo.submenu = None
    
    return menuInfo

#------------------------------------------------------------------------------

def ClickButton( hwnd ) :
    """Simulates a single mouse click on a button

    Arguments :
    hwnd    Window handle of the required button.

    Usage example :  okButton = FindControl( fontDialog,
                                             wantedClass="Button",
                                             wantedText="OK" )
                    ClickButton( okButton )
    """
    _SendNotifyMessage( hwnd, win32con.BN_CLICKED )

#------------------------------------------------------------------------------

def ClickStatic( hwnd ) :
    """Simulates a single mouse click on a static [ something ].

    Arguments :
    hwnd    - Window handle of the required static.

    Usage example :  TODO
    """
    _SendNotifyMessage( hwnd, win32con.STN_CLICKED )

#------------------------------------------------------------------------------

def DoubleClickStatic( hwnd ) :
    """Simulates a double mouse click on a static

    Arguments :
    hwnd    Window handle of the required static.

    Usage example :  TODO
    """
    _SendNotifyMessage( hwnd, win32con.STN_DBLCLK )

#------------------------------------------------------------------------------

def GetComboboxItems( hwnd ) :
    """Returns the items in a combo box control.

    Arguments :
    hwnd            Window handle for the combo box.

    Returns :        Combo box items.

    Usage example :  fontCombo = FindControl( fontDialog, wantedClass="ComboBox" )
                    fontComboItems = GetComboboxItems( fontCombo )
    """
    
    return _GetMultipleWindowValues( hwnd,
                                     getCountMessage=win32con.CB_GETCOUNT,
                                     getValueMessage=win32con.CB_GETLBTEXT )

#------------------------------------------------------------------------------

def SelectComboboxItem( hwnd, item ) :
    """Selects a specified item in a Combo box control.

    Arguments :
    hwnd            Window handle of the required combo box.
    item            The reqired item. Either an index, of the text of the
                    required item.

    Usage example :  fontComboItems = GetComboboxItems( fontCombo )
                    SelectComboboxItem( fontCombo,
                                       random.choice( fontComboItems ) )
    """
    try : # item is an index Use this to select
        0 + item
        win32gui.SendMessage( hwnd, win32con.CB_SETCURSEL, item, 0 )
        _SendNotifyMessage( hwnd, win32con.CBN_SELCHANGE )
    except TypeError : # Item is a string - find the index, and use that
        items = GetComboboxItems( hwnd )
        itemIndex = items.index( item )
        SelectComboboxItem( hwnd, itemIndex )

#------------------------------------------------------------------------------

def GetListboxItems( hwnd ) :
    """Returns the items in a list box control.

    Arguments :
    hwnd            Window handle for the list box.

    Returns :        List box items.

    Usage example :  TODO
    """
    
    return _GetMultipleWindowValues( hwnd,
                                     getCountMessage=win32con.LB_GETCOUNT,
                                     getValueMessage=win32con.LB_GETTEXT )

#------------------------------------------------------------------------------

def SelectListboxItem( hwnd, item ) :
    """Selects a specified item in a list box control.

    Arguments :
    hwnd            Window handle of the required list box.
    item            The reqired item. Either an index, of the text of the
                    required item.

    Usage example :  TODO
    """
    try : # item is an index Use this to select
        0 + item
        win32gui.SendMessage( hwnd, win32con.LB_SETCURSEL, item, 0 )
        _SendNotifyMessage( hwnd, win32con.LBN_SELCHANGE )
    except TypeError : # Item is a string - find the index, and use that
        items = GetListboxItems( hwnd )
        itemIndex = items.index( item )
        SelectListboxItem( hwnd, itemIndex )
                                    
#------------------------------------------------------------------------------

def GetEditText( hwnd ) :
    """Returns the text in an edit control.

    Arguments :
    hwnd            Window handle for the edit control.

    Returns         Edit control text lines.

    Usage example :  pprint.pprint( GetEditText( editArea ) )
    """
    return _GetMultipleWindowValues( hwnd,
                                    getCountMessage=win32con.EM_GETLINECOUNT,
                                    getValueMessage=win32con.EM_GETLINE )
    
#------------------------------------------------------------------------------

def SetEditText( hwnd, text, append=False ) :
    """Set an edit control's text.
    
    Arguments :
    hwnd            The edit control's hwnd.
    text            The text to send to the control. This can be a single
                    string, or a sequence of strings. If the latter, each will
                    be become a a seperate line in the control.
    append          Should the new text be appended to the existing text?
                    Defaults to False, meaning that any existing text will be
                    replaced. If True, the new text will be appended to the end
                    of the existing text.
                    Note that the first line of the new text will be directly
                    appended to the end of the last line of the existing text.
                    If appending lines of text, you may wish to pass in an
                    empty string as the 1st element of the 'text' argument.

    Usage example :  print "Enter various bits of text."
                    SetEditText( editArea, "Hello, again!" )
                    time.sleep( .5 )
                    SetEditText( editArea, "You still there?" )
                    time.sleep( .5 )
                    SetEditText( editArea, ["Here come", "two lines!"] )
                    time.sleep( .5 )
                    
                    print "Add some..."
                    SetEditText( editArea, ["", "And a 3rd one!"], append=True )
                    time.sleep( .5 )"""
    
    # Ensure that the text var contains text and then make it into a list.        
    try :
        text + ''
        text = [text]
    except TypeError :
        pass

    # Set the current selection range, depending on append flag
    if append :
        win32gui.SendMessage( hwnd, win32con.EM_SETSEL, -1, 0 )
    else :
        win32gui.SendMessage( hwnd, win32con.EM_SETSEL, 0, -1 )
                             
    # Send the text
    win32gui.SendMessage( hwnd, win32con.EM_REPLACESEL, True, os.linesep.join( text ) )

#------------------------------------------------------------------------------

def _GetMultipleWindowValues( hwnd, getCountMessage, getValueMessage ) :
    """A common pattern in the Win32 API is that in order to retrieve a
    series of values, you use one message to get a count of available
    items, and another to retrieve them. 
    
    This internal utility function performs the common processing for this pattern.

    Arguments :
    hwnd                Window handle for the window for which items should be
                        retrieved.
    getCountMessage     Item count message.
    getValueMessage     Value retrieval message.

    Returns :            Retrieved items."""
    result = []
    
    VALUE_LENGTH = 256
    bufferlength_int  = struct.pack( 'i', VALUE_LENGTH ) # This is a C style int.
    
    valuecount = win32gui.SendMessage( hwnd, getCountMessage, 0, 0 )
    for itemIndex in range( valuecount ) :
        valuebuffer = array.array( 'c',
                                  bufferlength_int +
                                  " " * ( VALUE_LENGTH - len( bufferlength_int ) ) )
        valueLength = win32gui.SendMessage( hwnd,
                                           getValueMessage,
                                           itemIndex,
                                           valuebuffer )
        result.append( valuebuffer.tostring()[ :valueLength] )
    #end for
    
    return result

#end _GetMultipleWindowValues def

#------------------------------------------------------------------------------

def _BuildWinLong( high, low ) :
    """Build a windows long parameter from high and low words.
    
    See http ://support.microsoft.com/support/kb/articles/q189/1/70.asp
    """
    
    # return ( ( high << 16 ) | low )
    return int( struct.unpack( '>L',
                               struct.pack( '>2H',
                               high,
                               low ) ) [0] )
        
#------------------------------------------------------------------------------

def _SendNotifyMessage( hwnd, nofifyMessage ) :
    """ Send a notify message to a control.
    """
    
    win32gui.SendMessage( win32gui.GetParent( hwnd ),
                          win32con.WM_COMMAND,
                          _BuildWinLong( nofifyMessage,
                          win32api.GetWindowLong( hwnd,
                          win32con.GWL_ID ) ),
                         hwnd )

#------------------------------------------------------------------------------

def _NormaliseText( controlText ) :
    """Remove '&' characters, and lower case.
    Useful for matching control text.
    """
    return controlText.lower().replace( '&', '' )

#------------------------------------------------------------------------------

def _FindNamedSubmenu( hMenu, hMenuItemCount, submenuName ) :
    """ Find the index number of a menu's submenu with a specific name.
    """
    
    for submenuIndex in range( hMenuItemCount ) :
        submenuInfo = GetMenuInfo( hMenu, submenuIndex )
        if _NormaliseText( submenuInfo.name ).startswith( _NormaliseText( submenuName ) ) :
            return submenuIndex, submenuInfo.submenu, submenuInfo.itemCount
            
    raise WinGuiAutoError( "No submenu found for hMenu=" +
                          repr( hMenu ) +
                          ", hMenuItemCount=" +
                          repr( hMenuItemCount ) +
                          ", submenuName=" +
                          repr( submenuName ) )

#------------------------------------------------------------------------------                
                              
class Bunch( object ) :
    """See http ://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/52308"""
    
    def __init__( self, **kwds ) :
        self.__dict__.update( kwds )
        
    def __str__( self ) :
        state = ["%s=%r" % ( attribute, value )
                 for ( attribute, value )
                 in self.__dict__.items()]
        return '\n'.join( state )
    
#------------------------------------------------------------------------------

class WinGuiAutoError( Exception ) :
    pass
    
#------------------------------------------------------------------------------
