#!/usr/bin/env python
#
# wxGameCanvas.py
#
# version 0.1, 7/14/2003
#
# To install:
#  Put in your site-packages folder or the same folder as your own source files.
#
# To import:
#  from wxGameCanvas import wxGameCanvas, InvokeLater, InvokeAndWait
#
# Instructions (see below and the included demo file)
#
# NOTE!!! Don't import pygame at the top of your file, import it in the "run" method of your
#         wxGameCanvas subclass, see below.
#
# Contact etd1 at edtechdev.org if you have any bug fixes or questions or additions, etc.
#
# Author: DougHolton
# License: wxPython license
#
# Thanks:
# InvokeLater/InvokeAndWait code adapted from bittorrent code and Robin Dunn's wxCallAfter.
# wxCallAfter does exactly the same thing as InvokeLater.
# wxGameCanvas adapted from http://wiki.wxpython.org/index.cgi/IntegratingPyGame
#
# HISTORY
# Version 0.1: 7/14/2003 initial release
#   - I've tested this on Windows, and tested resizing with LayoutAnchors and wxSplitterWindow resizing.
#


## INSTRUCTIONS:
## Converting standalone pygame code to wxpython code (with wxGameCanvas):
##
##    1. Create a subclass of wxGameCanvas that implements a "run" method where pygame drawing code goes.
##           Add this wxGameCanvas subclass to your wxFrame to show the game panel.  Your run method is
##           called right after the window first appears.  Do not do any pygame stuff before then.
##    2. Do NOT import pygame (or any other python files with pygame code) at the top of the file,
##           but rather at the beginning of the "run" method.  See the demo for an example.
##    3. Anytime you need to call other controls in the wxPython window from your "run" method, use
##          InvokeLater (or wxCallAfter) or InvokeAndWait.  If conversely something in the wxPython main thread
##          needs to tell your pygame drawing thread something, you might set boolean flags that your
##          "run" method checks for.  3 such flags are already built in:
##    4. self.running is the flag for stopping drawing.  Make your while loop in the "run" method like:
##          "while self.running:"
##    5. self.paused is the flag for pausing drawing.  Call self.handletasks() for convenience.
##    6. self.needresize is the flag for resizing.  self.handletasks() also handles resizing too.
##    7. You might remove any pygame fullscreen code, and just make the wxPython window fullscreen instead.
##          Remember the size of the drawing area may change, check self.width and self.height.
##    8. The main drawing surface is named self.surface.  If you are reusing pygame code that uses a
##          different name, try adding a line like "srf = self.surface" in your "run" method, or else
##          you can override the handleresize method.
##    9. If you are using a different drawing engine than pygame, you can override the initialize() method,
##          and override or just don't call handletasks() and the other convenience methods.


from wxPython.wx import *
from threading import Thread, Event

wxEVT_INVOKE = None
firstinvoke = True

def EVT_INVOKE(win, func):
    if win is None:
        win = wxGetApp()
        assert win, 'No wxApp created yet'
    global wxEVT_INVOKE
    if wxEVT_INVOKE is None:
        wxEVT_INVOKE = wxNewEventType()
    win.Connect(-1, -1, wxEVT_INVOKE, func)

class InvokeEvent(wxPyEvent):
    def __init__(self, func, args, kwargs):
        wxPyEvent.__init__(self)
        self.SetEventType(wxEVT_INVOKE)
        self.func = func
        self.args = args
        self.kwargs = kwargs
        self.evt = Event()
        self.result = None
    
def InvokeLater(func, *args, **kwargs):
    """ Use this method to update the wxPython GUI from the PyGame thread (your run() method).
        Any calls to wxPython controls need to be done on the main thread, not the pygame
        thread, hence the use of wxPostEvent.  Another alternative is to just call wxCallAfter
        which does the same thing.

        You may need to wrap the arguments to the function you want to call with brackets.
        Examples:
            InvokeLater(self.UpdateControlValues, param1, param2, param3)
            InvokeLater(self.status.SetLabel, 'Time elapsed: %s'(elapsedtime))

        If you need to wait for the function to return, like when you want to ask the user
        to select a file, use InvokeAndWait instead.            
    """
    global firstinvoke
    if firstinvoke:
        EVT_INVOKE(None, OnInvoke)
        firstinvoke = False
    wxPostEvent(wxGetApp(), InvokeEvent(func, args, kwargs))

def InvokeAndWait(func, *args, **kwargs):
    """Similar to InvokeLater, except it blocks your thread's execution until
    the function returns back from the main thread to your drawing thread.
    Use this for calling wxPython functions that you need to return data to you,
    such as having the user select a file or choose a color.
    Example:
        file = InvokeAndWait(self.OnChooseFile, [default, size, dir])
    """
    global firstinvoke
    if firstinvoke:
        EVT_INVOKE(None, OnInvoke)
        firstinvoke = False
    invoke = InvokeEvent(func, args, kwargs)
    wxPostEvent(wxGetApp(), invoke)
    invoke.evt.wait()
    return invoke.result

def OnInvoke(event):
    """This method is fired by wxPostEvent in the main (wxPython) thread.
    See InvokeLater and InvokeAndWait.
    """
    event.result = apply(event.func, event.args, event.kwargs)
    if event.evt: event.evt.set()




###############################################################################



class wxGameCanvas(wxWindow):

    def __init__(self, parent,id = -1, pos = wxDefaultPosition, size = wxDefaultSize):
        
        wxWindow.__init__(self, parent, id, pos=pos, size=size)

        self.drawthread = Thread(None, self.run, self)
        self.firsttime = True
        self.running = False
        self.paused = False
        self.width, self.height = size
        self.needresize = True
        self.finished = False
        #self.closemessage = "Please wait while the program is stopping."
        self.timeout = 2.0 #how many seconds max to wait for drawing thread to stop.
        self.surface = None
        self.oldsurface = None
        self.scaleonpause = False #If you set to True, if you resize while paused, the image grows or shrinks

        self.width, self.height = 0,0
        self.oldwidth, self.oldheight = 0,0

        EVT_ERASE_BACKGROUND(self, self.OnEraseBackground)
        EVT_SIZE(self, self.OnReSize)
        EVT_PAINT(self, self.OnPaint)
        EVT_CLOSE(self, self.OnClose) #this never gets called, so shutdown of thread happens in __del__ instead

    def __del__(self):
        self.stop()
        #self.cleanup()
        wxWindow.__del__(self)

    def start(self):
        "Start the thread, which fires your run() method"
        if self.running or self.drawthread.isAlive():
            return #already started
        self.running = True
        self.paused = False
        if not self.drawthread: #maybe we are restarting a thread
            self.drawthread = Thread(None, self.run, self)
        self.drawthread.start()

    def stop(self):
        """This stops the drawing thread, you have to check self.running in your while loop.
            Otherwise stop() forces the thread to stop.
        """
        if not self.drawthread:
            return
        if (not self.running) and (not self.drawthread.isAlive()): #already stopped
            return

        #busy = wxBusyInfo(self.closemessage)
        
        self.running = False
        self.paused = False

        import time
        t=0
        while self.drawthread.isAlive(): #first we'll see if your run method stops itself.
            if t > self.timeout: break
            wxYield()
            time.sleep(0.1)
            t += 0.1
            
        self.drawthread.join(self.timeout) #force the thread to stop if it hasn't already
        self.drawthread = None
        self.finished = true

    def pause(self):
        """A convenience method.  Call self.handletasks() in your while loop to handle pausing.
            Call self.unpause() to unpause.
        """
        self.paused = True

    def unpause(self):
        self.paused = False

    def OnEraseBackground(self, event):
        pass

    def OnClose(self, event):
        """ This apparently isn't called at all.  So the real shutdown happens in __del__ """
        self.stop()
        event.Skip()

    def OnPaint(self, event):
        #print "PAINT EVENT IN wxGameCanvas"
        self.oldwidth, self.oldheight = self.width, self.height
        self.width, self.height = self.GetClientSize().width, self.GetClientSize().height
        if not (self.width > 0 and self.height > 0):
            event.Skip()
            return
        dc = wxPaintDC(self)
        if self.firsttime:
            self.needresize = True
            self.initialize()
            self.firsttime = False
            return
        elif (self.width != self.oldwidth) or (self.height != self.oldheight):
            self.needresize = True
            self.handleresize() #might comment this out

    def OnReSize(self, event):
        """Initialize pygame AFTER window is created, in the first resized event.
        
            Afterward, just set the needresize flag to let your pygame drawing loop know
            it needs to resize.
        """        
        #print "RESIZE EVENT IN wxGameCanvas"
        self.oldwidth, self.oldheight = self.width, self.height
        self.width, self.height = event.GetSize().width, event.GetSize().height
        if self.firsttime:
            if not (self.width > 0 and self.height > 0):
                event.Skip()
                return
            self.needresize = True
            self.initialize()
            self.firsttime = False
        else:
           self.needresize = True

    def run(self):
        """This method should be implemented by subclass and is called by the drawing thread. Example basic code:
        
            import pygame
            while self.running:
                self.handletasks()
                your drawing code...

        """
        raise NotImplementedError('Please define a run() method in your wxGameCanvas subclass')

    #
    # The rest of these methods are pygame specific.  If you are not using pygame, the main
    # method you need to override is initialize()
    #

    def setwindowhandle(self):
        #you can delete this if you are not using pygame/sdl for rendering:
        import os, sys
        os.environ['SDL_WINDOWID'] = str(self.GetHandle())
        if sys.platform == "win32":
            os.environ['SDL_VIDEODRIVER'] = 'windib'

    def initialize(self):
        """Called the first time pygame is needed to be initialized.  Override this method if necessary. """
        self.setwindowhandle()
        import pygame
        if pygame.display.get_init(): #uh oh, pygame initialized too early
            print "wxGameCanvas.initialize(): PyGame already initialized. Don't import pygame before your run method starts."
            pass            
        pygame.display.init()
        self.handleresize()
        self.start()

##    def cleanup(self):
##        """ Called when the wxGameCanvas is about to be deleted from memory. """
##        import pygame
##        pygame.quit()

    def handleresize(self):
        """Handle resize requests.  Resets self.surface. """
        if self.needresize:
            import pygame
            if (self.width < 1) or (self.height < 1): return #pygame will throw an error otherwise
            
            self.surface = pygame.display.set_mode((self.width,self.height))

            if self.paused and self.surface and self.oldsurface: #a couple of hacks to handle resizing while paused
                if self.scaleonpause: #this hack takes the old image and resizes it and draws it (can get ugly)
                    tempsurface = pygame.transform.scale(self.oldsurface,(self.width,self.height))
                    self.surface.blit(tempsurface,(0,0))
                else:
                    #this hack just draws the old image as is.  But if the new size is larger, it first fills the background.
                    if (self.width > self.oldwidth) or (self.height > self.oldheight):
                        #This is a hack to guess the background color:
                        fillcolor = self.oldsurface.get_at((0,0))
                        self.surface.fill(fillcolor)
                    self.surface.blit(self.oldsurface,(0,0))
                pygame.display.flip()
            self.needresize = False
            
    def handlepause(self):
        import pygame
        if self.paused and self.surface: #store a copy of the drawing image in case resize during pause.
            self.oldsurface = pygame.transform.scale(self.surface, (self.width,self.height))
        while self.paused: #this could be a while
            self.handleresize() #the user might resize while paused
            self.handlequit() #handle quit events regardless if paused
            pygame.time.wait(100)
        self.oldsurface = None

    def handlequit(self):
        """Handle quit events in pygame """
        import pygame
        from pygame.locals import QUIT
        events = pygame.event.get()
        for e in events:
            if e.type == QUIT:
                self.stop()        

    def handletasks(self, handlequit=False):
        """Handle resizing, pausing, etc. for your animation loop.  The optional handlequit parameter (default is False)
        will handle the quit event too.  Don't use it if you are handling events yourself. """
        #self.setwindowhandle() #Maybe needed if more than one wxGameCanvas?  No SDL equivalent to aglSetCurrentContext.
        if handlequit:
            self.handlequit()
        if self.needresize:
            self.handleresize()
        if self.paused:
            self.handlepause()            


        
        

    
