# buffered_with_controls.py

import random
import wx
import wx.lib.buttons  as  buttons   

# In ms. 
REFRESH_RATE = 1000 

# class MyBufferedWindow
# class MyDrawWindow
# class MyFrame
# class MyApp

#-------------------------------------------------------------------------------

class MyBufferedWindow(wx.Window):
    def __init__(self, parent, id,
                 pos=wx.DefaultPosition,
                 size=wx.DefaultSize,
                 style=wx.NO_FULL_REPAINT_ON_RESIZE):
        wx.Window.__init__(self, parent, id, pos, size,
                           style | wx.CLIP_CHILDREN)

        self.Bind(wx.EVT_PAINT, self.OnPaint)
        self.Bind(wx.EVT_SIZE, self.OnSize)
        
        # OnSize called to make sure the buffer is initialized.
        # This might result in OnSize getting called twice on some
        # platforms at initialization, but little harm done.
        self.OnSize(None)

    #-------------------------------------------------------------------
        
    def Draw(self, dc):
        """
        Just here as a place holder.
        This method should be over-ridden when sub-classed.
        """
        
        pass


    def OnPaint(self, event):
        """
        All that is needed here is to draw the buffer to screen.
        """
        
        dc = wx.BufferedPaintDC(self, self._Buffer)


    def OnSize(self, event):
        """
        ...
        """
        
        # The Buffer init is done here, to make sure the 
        # buffer is always the same size as the Window.
        self.Width, self.Height = self.GetClientSize()

        # Make new off screen bitmap : this bitmap will always have the
        # current drawing in it, so it can be used to save the image to
        # a file, or whatever.
        self._Buffer = wx.Bitmap(self.Width, self.Height)
        self.UpdateDrawing()


    def SaveToFile(self,FileName, FileType):
        """
        This will save the contents of the buffer
        to the specified file. See the wxWindows docs
        for wx.Bitmap::SaveFile for the details.
        """
        
        self._Buffer.SaveFile(FileName, FileType)


    def UpdateDrawing(self):
        """
        This would get called if the drawing needed to change, for whatever reason.
        The idea here is that the drawing is based on some data generated
        elsewhere in the system. If that data changes, the drawing needs to
        be updated.
        """
        
        dc = wx.MemoryDC()
        dc.SelectObject(self._Buffer)
        self.Draw(dc)
        # This forces a Paint event, so the screen gets updated.
        self.Refresh() 
        # If it's not getting updated fast enough, this should force it. 
        # self.Update() 

#-------------------------------------------------------------------------------

class MyDrawWindow(MyBufferedWindow):
    def __init__(self, parent, id=-1):
        """
        Any data the Draw() function needs must be initialized before
        calling MyBufferedWindow.__init__, as it will call the Draw
        function.
        """
        
        self.DrawData = {}
        
        MyBufferedWindow.__init__(self, parent, id)

        #------------

        # Simplified init method.        
        self.BtnNormal()
        self.BtnBitmap()
        self.CheckB()

    #-------------------------------------------------------------------
        
    def Draw(self, dc):
        """
        ...
        """
        
        dc.SetBackground(wx.Brush("Gray"))
        dc.Clear() # Make sure you clear the bitmap !

        # Here's the actual drawing code.
        for key,data in self.DrawData.items():
            if key == "Rectangles":
                dc.SetBrush(wx.YELLOW_BRUSH)
                dc.SetPen(wx.Pen("VIOLET", 4))
                for r in data:
                    dc.DrawRectangle(*r)

            elif key == "Ellipses":
                dc.SetBrush(wx.Brush("GREEN"))
                dc.SetPen(wx.Pen("CADET BLUE", 2))
                for r in data:
                    dc.DrawEllipse(*r)

            elif key == "Polygons":
                dc.SetBrush(wx.Brush("SALMON"))
                dc.SetPen(wx.Pen("VIOLET RED", 4))
                for r in data:
                    dc.DrawPolygon(r)


    def BtnNormal(self):
        """
        ...
        """
        
        self.btnNormal = wx.Button(self, 600, "normal btn", pos=(100, 100))
        self.Bind(wx.EVT_BUTTON, self.OnNormalButton, id=600)

        
    def OnNormalButton(self,event=None):
        """
        ...
        """
        
        print("Normal Button Clicked")


    def BtnBitmap(self):
        """
        ...
        """

        # Only bitmap buttons seam to work.        
        self.buttonTest2 = buttons.GenBitmapTextButton(self, 700, None,
                                                       "bitmap btn",
                                                       pos=(250, 250))
        self.Bind(wx.EVT_BUTTON, self.OnBitmapButton, id=700)


    def OnBitmapButton(self,event=None):
        """
        ...
        """
        
        print("Bitmap Button Clicked")


    def CheckB(self):
        """
        ...
        """
        
        self.checkB = wx.CheckBox(self, 800, "test", pos=(300, 50))

#-------------------------------------------------------------------------------
        
class MyFrame(wx.Frame):
    def __init__(self):
        wx.Frame.__init__(self, None, -1,
                          "Double Buffered with controls",
                         wx.DefaultPosition,
                         size=(420, 420),
                         style=wx.DEFAULT_FRAME_STYLE |
                               wx.NO_FULL_REPAINT_ON_RESIZE)

        #------------

        self.SetIcon(wx.Icon('wxwin.ico'))        
        self.SetMinSize((320, 320))
        
        #------------

        # Delete flicker.        
        if wx.Platform == "__WXMSW__":
            self.SetDoubleBuffered(True)

        #------------
        
        # Set up the menuBar.
        menuBar = wx.MenuBar()

        file_menu = wx.Menu()

        file_menu.Append(500, "E&xit", "Terminate the program")
        self.Bind(wx.EVT_MENU, self.OnQuit, id=500)
        menuBar.Append(file_menu, "&File")

        draw_menu = wx.Menu()
        
        draw_menu.Append(400, "&New Drawing","Update the Drawing Data")
        self.Bind(wx.EVT_MENU,self.NewDrawing, id=400)
        
        draw_menu.Append(300, "&Save Drawing\tAlt-I", "")
        self.Bind(wx.EVT_MENU, self.SaveToFile, id=300)
        
        menuBar.Append(draw_menu, "&Draw")

        draw_menu.Append(200, "&Start Periodic Update\tF5")
        self.Bind(wx.EVT_MENU, self.StartUpdate, id=200)
        
        self.SetMenuBar(menuBar)

        draw_menu.Append(100, "&Stop Periodic Update\tF6")
        self.Bind(wx.EVT_MENU, self.StopUpdate, id=100)

        self.SetMenuBar(menuBar)

        #------------
        
        self.Window = MyDrawWindow(self)

        #------------
        
        # Now set up the timer.
        self.timer = wx.Timer(self)
        self.Bind(wx.EVT_TIMER, self.Notify)

    #-------------------------------------------------------------------

    def StartUpdate(self, event):
        """
        ...
        """

        # Time between events (in milliseconds).      
        self.timer.Start(1000)  


    def StopUpdate(self, event=None):
        """
        ...
        """
        
        self.timer.Stop()


    def Notify(self, event):
        """
        ...
        """
        
        self.NewDrawing(None)


    def OnQuit(self, event):
        """
        ...
        """
        
        self.Close(True)


    def NewDrawing(self, event):
        """
        ...
        """
        
        self.Window.DrawData = self.MakeNewData()
        self.Window.UpdateDrawing()


    def SaveToFile(self, event):
        """
        ...
        """
        
        dlg = wx.FileDialog(self, "Choose a file name to save the image as a PNG",
                            defaultDir="",
                            defaultFile="",
                            wildcard="*.png",
                            style=wx.FD_SAVE)
        if dlg.ShowModal() == wx.ID_OK:
            self.Window.SaveToFile(dlg.GetPath(), wx.BITMAP_TYPE_PNG)
        dlg.Destroy()


    def MakeNewData(self):
        """
        This method makes some random data to draw things with.
        """
        
        MaxX, MaxY = self.Window.GetClientSize()
        DrawData = {}

        # Make some random rectangles.
        l = []
        for i in range(2):
            w = random.randint(1, int(MaxX/2))
            h = random.randint(1, int(MaxY/2))
            x = random.randint(1, int(MaxX-w))
            y = random.randint(1, int(MaxY-h))
            l.append( (x,y,w,h) )
        DrawData["Rectangles"] = l

        return DrawData

#-------------------------------------------------------------------------------

class MyApp(wx.App):
    def OnInit(self):
        # Called so a PNG can be saved.
        # wx.InitAllImageHandlers() 
        frame = MyFrame()
        frame.Show(True)

        # Initialize a drawing.
        # It doesn't seem like this should be here, but the Frame does
        # not get sized until Show() is called, so it doesn't work if
        # it is put in the __init__ method.
        frame.NewDrawing(None)

        self.SetTopWindow(frame)

        return True

#-------------------------------------------------------------------------------

if __name__ == "__main__":
    app = MyApp(False)
    app.MainLoop()
