# sample_two.py

import sys
import os.path
import wx
import wx.stc as stc   # StyledTextCtrl

# class MyFrame
# class MyApp

#-------------------------------------------------------------------------------

# This is how you pre-establish a file filter so that the 
# dialog only shows the extension(s) you want it to.

wildcard = "Text (*.txt)|*.txt|"        \
           "Config (*.cfg)|*.cfg|"      \
           "Python (*.py)|*.py|"        \
           "Python (*.pyw)|*.pyw|"      \
           "All (*.*)|*.*"

#-------------------------------------------------------------------------------

class MyFrame(wx.Frame):
    def __init__(self, filename="noname.txt"):
        super(MyFrame, self).__init__(None, size=(400, 300))
        
        #------------

        # Return icons folder.
        self.icons_dir = wx.GetApp().GetIconsDir()

        #------------
        
        self.filename = filename
        self.dirname  = "."
        self.flagDirty = False

        #------------
        
        # Simplified init method.
        self.SetProperties()
        self.CreateMenuBar()
        self.CreateStyledTextControl()
        self.CreateStatusBar()
        self.BindEvents()

    #---------------------------------------------------------------------------

    def SetProperties(self):
        """
        ...
        """ 

        self.SetTitle()

        #------------
        
        frameIcon = wx.Icon(os.path.join(self.icons_dir,
                                         "icon_wxWidgets.ico"),
                            type=wx.BITMAP_TYPE_ICO)
        self.SetIcon(frameIcon)


    def SetTitle(self):
        """
        ...
        """
        
        # MyFrame.SetTitle overrides wx.Frame.SetTitle,  
        # so we have to call it using super :
        super(MyFrame, self).SetTitle("Editor %s" % self.filename)

        
    def CreateMenuBar(self):
        """
        Create menu and menu bar.
        """ 

        menuBar = wx.MenuBar()

        #------------
        
        menu_File = wx.Menu()
        menu_Edit = wx.Menu()
        menu_About = wx.Menu()
        
        #------------
        #------------
        
        self.menu_File_Open = menu_File.Append(wx.ID_OPEN,
                                               "&Open" + "\t" + "Ctrl+O",
                                               "Open a new file.")        
        #------------
        
        menu_File.AppendSeparator()

        #------------
        
        self.menu_File_Save = menu_File.Append(wx.ID_SAVE,
                                               "&Save" + "\t" + "Ctrl+S",
                                               "Save the current file.")
        self.menu_File_SaveAs = menu_File.Append(wx.ID_SAVEAS,
                                                 "&Save as" + "\t" + "Ctrl+Shift+S",
                                                 "Save the file under a different name.")

        #------------
        
        menu_File.AppendSeparator()

        #------------
        
        self.menu_File_Close = menu_File.Append(wx.ID_EXIT,
                                                "&Exit" + "\t" + "Ctrl+X",
                                                "Exit the program.")
        #------------
        
        self.menu_Edit_Cut = menu_Edit.Append(wx.ID_CUT,
                                              "&Cut" + "\t" + "Ctrl+X",
                                              "Cut")
        self.menu_Edit_Copy = menu_Edit.Append(wx.ID_COPY,
                                               "&Copy" + "\t" + "Ctrl+C",
                                               "Copy")
        self.menu_Edit_Paste = menu_Edit.Append(wx.ID_PASTE,
                                                "&Paste" + "\t" + "Ctrl+V",
                                                "Paste")
        #------------
        
        menu_Edit.AppendSeparator()

        #------------
        
        self.menu_Edit_Undo = menu_Edit.Append(wx.ID_UNDO,
                                               "&Undo" + "\t" + "Ctrl+Z",
                                               "Undo")
        self.menu_Edit_Redo = menu_Edit.Append(wx.ID_REDO,
                                               "&Redo" + "\t" + "Ctrl+Shift+Z",
                                               "Redo")

        #------------

        self.menu_About_Info = menu_About.Append(wx.ID_ABOUT,
                                                 "&About" + "\t" + "Ctrl+I",
                                                 "Information about this program.")
        
        #------------
        #------------
        
        menuBar.Append(menu_File, "File")
        menuBar.Append(menu_Edit, "Edit")
        menuBar.Append(menu_About, "About")
        
        #------------

        self.SetMenuBar(menuBar)
        

    def CreateStyledTextControl(self):
        """
        ...
        """
        
        self.St_TextCtrl = stc.StyledTextCtrl(self)
        self.St_TextCtrl.SetWindowStyle(self.St_TextCtrl.GetWindowStyle() | wx.DOUBLE_BORDER)
        self.St_TextCtrl.StyleSetSpec(stc.STC_STYLE_DEFAULT, "size:10,face:Courier New")
        self.St_TextCtrl.SetWrapMode(stc.STC_WRAP_WORD) 

        #------------
        
        layout = wx.BoxSizer(wx.HORIZONTAL)
        layout.Add(self.St_TextCtrl, proportion=1, border=0, flag=wx.ALL|wx.EXPAND)
        self.SetSizer(layout)

        #------------
        
        self.St_TextCtrl.Bind(stc.EVT_STC_CHANGE, self.OnChangeTxtCtrl)


    def BindEvents(self):
        """
        ...
        """

        self.Bind(wx.EVT_MENU, self.OnOpen, self.menu_File_Open)
        self.Bind(wx.EVT_MENU, self.OnSave, self.menu_File_Save)
        self.Bind(wx.EVT_MENU, self.OnSaveAs, self.menu_File_SaveAs)
        self.Bind(wx.EVT_MENU, self.OnCloseMe, self.menu_File_Close)
        self.Bind(wx.EVT_MENU, self.OnCut, self.menu_Edit_Cut)
        self.Bind(wx.EVT_MENU, self.OnCopy, self.menu_Edit_Copy)
        self.Bind(wx.EVT_MENU, self.OnPaste, self.menu_Edit_Paste)
        self.Bind(wx.EVT_MENU, self.OnUndo, self.menu_Edit_Undo)
        self.Bind(wx.EVT_MENU, self.OnRedo, self.menu_Edit_Redo)
        self.Bind(wx.EVT_MENU, self.OnAbout, self.menu_About_Info)
        
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)

        
    def OnChangeTxtCtrl(self, event):
        """
        ...
        """
        
        lines = self.St_TextCtrl.GetLineCount()
        width = self.St_TextCtrl.TextWidth(stc.STC_STYLE_LINENUMBER, str(lines) + " ")
        self.St_TextCtrl.SetMarginWidth(0, width)

        self.flagDirty = True


    def DefaultFileDialogOptions(self):
        """
        Return a dictionary with file dialog options that can be
        used in both the save file dialog as well as in the open
        file dialog.
        """

        return dict(message="Choose a file :",
                    defaultDir=self.dirname,
                    wildcard=wildcard)


    def AskUserForFilename(self, **dialogOptions):
        """
        ...
        """

        dialog = wx.FileDialog(self, **dialogOptions)
        
        if dialog.ShowModal() == wx.ID_OK:
            userProvidedFilename = True
            self.filename = dialog.GetFilename()
            self.dirname = dialog.GetDirectory()
            # Update the window title with the new filename.
            self.SetTitle() 
        else:
            userProvidedFilename = False
            
        dialog.Destroy()
        
        return userProvidedFilename


    def OnOpen(self, event):
        """
        Open file.
        """
        
        if self.AskUserForFilename(style=wx.FD_OPEN,
                                   **self.DefaultFileDialogOptions()):
            file = open(os.path.join(self.dirname, self.filename), 'r', encoding='utf-8')
            self.St_TextCtrl.SetValue(file.read())
            file.close()
        
           
    def OnSave(self, event):
        """
        Save file.
        """

        with open(os.path.join(self.dirname, self.filename), 'w', encoding='utf-8') as file:
            file.write(self.St_TextCtrl.GetValue())
        
        
    def OnSaveAs(self, event):
        """
        Save file as.
        """

        if self.AskUserForFilename(defaultFile=self.filename, style=wx.FD_SAVE,
                                   **self.DefaultFileDialogOptions()):
            self.OnSave(event)


    def OnCut(self, event):
        """
        ...
        """
        
        self.St_TextCtrl.Cut()


    def OnCopy(self, event):
        """
        ...
        """
        
        self.St_TextCtrl.Copy()


    def OnPaste(self, event):
        """
        ...
        """
        
        self.St_TextCtrl.Paste()


    def OnUndo(self, event):
        """
        ...
        """
        
        self.St_TextCtrl.Undo()


    def OnRedo(self, event):
        """
        ...
        """
        
        self.St_TextCtrl.Redo()


    def OnAbout(self, event):
        """
        About dialog.
        """

        dialog = wx.MessageDialog(self,
                                  "A sample StyledTextCtrl editor in wxPython.",
                                  "About sample editor",
                                  wx.OK)
        dialog.ShowModal()
        dialog.Destroy()


    def OnCloseMe(self, event):
        """
        Close the main window.
        """

        self.Close(True)

        
    def OnCloseWindow(self, event):
        """
        Quit and destroy application.
        """
        
        self.Destroy()

#-------------------------------------------------------------------------------
      
class MyApp(wx.App):
    """
    ....
    """  
    def OnInit(self):

        #------------

        self.installDir = os.path.split(os.path.abspath(sys.argv[0]))[0]

        #------------

        frame = MyFrame()
        self.SetTopWindow(frame)
        frame.Show(True)
        
        return True

    #---------------------------------------------------------------------------

    def GetInstallDir(self):
        """
        Returns the installation directory for my application.
        """

        return self.installDir


    def GetIconsDir(self):
        """
        Returns the icons directory for my application.
        """

        icons_dir = os.path.join(self.installDir, "icons")
        return icons_dir
    
#-------------------------------------------------------------------------------

def main():
    app = MyApp(False)
    app.MainLoop()

#-------------------------------------------------------------------------------

if __name__ == "__main__" :
    main()
       
