# reversi.py

"""
wxPython Reversi game.

Author : Hatena
Link : https://yatt.hatenablog.jp/entry/20100129/1264791420
"""

import wx
from copy import deepcopy

WHITE = -1
BLANK = 0
BLACK = 1

# def Notify
# class Reversi
# class Frame

#---------------------------------------------------------------------------

def Notify(caption, message):
    dialog = wx.MessageDialog(None,
                              message=message,
                              caption=caption,
                              style=wx.OK)
    dialog.ShowModal()
    dialog.Destroy()

#---------------------------------------------------------------------------

class Reversi(object):
    def __init__(self):

        #------------
        
        self.board = [[0]*8 for x in range(8)]
        self.board[3][3] = WHITE
        self.board[4][4] = WHITE
        self.board[3][4] = BLACK
        self.board[4][3] = BLACK
        self.turns = 1              # Turn counter.
        self.current = BLACK        # Current player.
        self.nstone = [2, 60, 2]    # white, blank, black.
        self.nreversed = 0
        self.passed = None
        self.over = False

    #-----------------------------------------------------------------------
        
    def __getitem__(self, i):
        return self.board[i]


    def Stones(self, s):
        return self.nstone[1 + s]


    def Feverseline(self, x, y, dx, dy):
        if not (0<=x<8) or not (0<=y<8):
            return False
        
        elif self[x][y] == BLANK:
            return False
        
        elif self[x][y] == self.current:
            return True
        
        elif self[x][y] == -self.current:
            ret = self.Feverseline(x+dx, y+dy, dx, dy)
            if ret:
                self[x][y] *= -1 # Reverse.
                self.nreversed += 1
            return ret


    def Put(self, x, y, validation=False):
        if self.over or self[x][y] != BLANK:
            return False
        
        backup = self[x][y]
        self[x][y] = self.current
        self.nreversed = 0
        
        for dx in [-1, 0, 1]:
            for dy in [-1, 0, 1]:
                self.Feverseline(x+dx, y+dy, dx, dy)
                
        if self.nreversed == 0:
            self[x][y] = backup
            return False
        
        if validation:
            # Despress changing state but only self.board.
            return True
        
        self.nstone[1 + BLANK] -= 1
        self.nstone[1 + self.current] += self.nreversed + 1
        self.nstone[1 - self.current] -= self.nreversed
        self.turns += 1
        self.current *= -1 # Change turn.
        
        # Check game is over or not.
        for s in [WHITE, BLANK, BLACK]:
            if self.Stones(s) == 0:
                self.over = True
                return True
            
        for i in range(8):
            for j in range(8):
                if self.Available(i, j):
                    self.passed = None
                    return True
                
        self.passed = self.current
        self.current *= -1
        
        for i in range(8):
            for j in range(8):
                if self.Available(i, j):
                    return True
        self.current *= -1
        self.over = True       
        return True


    def Available(self, x, y):
        if self.over:
            return False

        # Backup board.
        backup = deepcopy(self.board)
        ret = self.Put(int(x), int(y), True)
        # Restore board.
        self.board = backup        
        return ret

#---------------------------------------------------------------------------
    
class Frame(wx.Frame):
    def __init__(self):
        style=wx.DEFAULT_FRAME_STYLE ^ wx.RESIZE_BORDER ^ wx.MAXIMIZE_BOX
        wx.Frame.__init__(self, None, -1, "Reversi", style=style)

        #------------

        self.SetIcon(wx.Icon(".\icons\wxwin.ico"))
      
        #------------
       
        # Panel.
        self.panel = wx.Panel(self)
        self.panel.Bind(wx.EVT_LEFT_DOWN, self.TryPut)
        self.panel.Bind(wx.EVT_PAINT, self.Refresh)
        
        #------------
        
        self.Newgame(None)

        #------------
        
        # Menubar.
        menu = wx.Menu()
        menu.Append(1, "New game")
        menu.AppendSeparator()
        menu.Append(2, "Quit")
        menubar = wx.MenuBar()
        menubar.Append(menu, "Menu")
        self.SetMenuBar(menubar)

        self.Bind(wx.EVT_MENU, self.Newgame, id=1)
        self.Bind(wx.EVT_MENU, self.Quit, id=2)

        #------------
        
        # Status bar.
        self.CreateStatusBar()

        #------------
        
        # Ban resize.
        # self.Bind(wx.EVT_SIZE, lambda e:e)

        #------------
        
        self.Show()
        
    #-----------------------------------------------------------------------
       
    def Quit(self, event):
        self.Close()

    
    def Newgame(self, event):
        # Initialize reversi and Refresh screen.
        self.reversi = Reversi()
        self.panel.Refresh()

    
    def TryPut(self, event):
        if self.reversi.over:
            return
        
        # Calculate coordinate from window coordinate.
        winx,winy = event.GetX(), event.GetY()
        w,h = self.panel.GetSize()
        x = winx / (w/8)
        y = winy / (h/8)
        
        if not self.reversi.Available(int(x), int(y)):
            return
        
        ret = self.reversi.Put(int(x), int(y))
        self.panel.Refresh()
       
        # If game is over then display dialog.
        if self.reversi.over:
            black = self.reversi.Stones(BLACK)
            white = self.reversi.Stones(WHITE)
            mes = "Black : %d\nWhite : %d\n" % (black, white)
            if black == white:
                mes += "** draw **"
            else:
                mes += "Winner : %s" % ["Black", "White"][black < white]
            Notify("Game is over !", mes)
            
        elif self.reversi.passed != None:
            Notify("Passing turn", "Pass")


    def Refresh(self, event):
        dc = wx.AutoBufferedPaintDCFactory(self.panel)
        dc = wx.GCDC(dc)
        self.SetStatusText("Current player is " + ["White", "Black"][self.reversi.current==BLACK])
        w,h = self.panel.GetSize()

        # Background.       
        dc.SetBrush(wx.Brush("#228b22"))
        dc.DrawRectangle(0,0,w,h)
        # Grid.
        dc.SetBrush(wx.Brush("black"))
        px,py = w/8,h/8
        
        for i in range(8):
            dc.DrawLine(i*px,0, i*px, h)
            dc.DrawLine(0, i*py, w, i*py)
        dc.DrawLine(w-1, 0, w-1, h-1)
        dc.DrawLine(0, h-1, w-1, h-1)
        
        # Stones.
        brushes = {WHITE: wx.Brush("white"), BLACK: wx.Brush("black")}
        for i in range(8):
            for j in range(8):
                c = self.reversi[i][j]
                if c != BLANK:
                    dc.SetBrush(brushes[c])
                    dc.DrawEllipse(i*px, j*py, px, py)
                    
                elif self.reversi.Available(i, j):
                    dc.SetBrush(wx.Brush("red"))
                    dc.DrawCircle(i*px+px/2, j*py+py/2, 3)
   
#---------------------------------------------------------------------------
    
def main():
    app = wx.App(False)
    frame = Frame()
    frame.SetSize((400, 400))
    app.MainLoop()

#---------------------------------------------------------------------------
    
if __name__ == '__main__':
    main()
    
