'''
    ToggleButtonBitmap.py

    A custom class On/Off clickable image, where you supply an Off image and an On image
    The images are scaled according to the size parameter you supply

    Events: EVT_TBB_ON_OFF          Generic event on binary change

            EVT_TBB_ON              The control changed to On

            EVT_TBB_OFF             The control changed to Off

    Event Notes:
        All events return GetValue() i.e.

            def OnOff(self,event):
                current_value = event.GetValue()

    Functions:
        GetValue()              Returns numeric value in the control On (True) = 1 Off (False) = 0

        SetValue(int)           Set numeric value in the control

        Enable(Bool)            Enable/Disable the control
                                On Disable the control value is frozen

        IsEnabled()             Is the control Enabled - Returns True/False

        SetBitmapOn(image)      Set the On image

        SetBitmapOff(image)     Set the Off image

        SetToolTip(str)         Set tooltip for the control

    Default Values:
        initial -       0 (False) Off

Author:     J Healey
Created:    31/01/2021
Copyright:  J Healey - 2021-2023
License:    GPL 3 or any later version
Version:    2.0.0
Email:      <rolfofsaxony@gmx.com>
Written & tested: Linux, Python 3.8.10

Change log:
version 2.0.0
    Adds Tooltips
    Adds OnDisabled and OffDisabled bitmaps
    Removes wx.EmptyBitmap as deprecated
    Uses wx.Bitmap.ConvertToDisabled() to generate disabled bitmaps
    Adds Disable function default True

Usage example:

import wx
import ToggleButtonBitmap as tbb
class Frame(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent, -1, "On Off button Demo")
        panel = wx.Panel(self)
        panel.SetBackgroundColour("#f0ffff")
        self.onoff = tbb.ToggleButtonBitmap(panel, -1, bitmapon="G1.png", bitmapoff="G2.png", size=(200,200), initial=0)
        self.Show()

app = wx.App()
frame = Frame(None)
app.MainLoop()
'''

import wx

#Events OnOff, On and Off
tbbEVT_ON_OFF = wx.NewEventType()
EVT_ON_OFF = wx.PyEventBinder(tbbEVT_ON_OFF, 1)
tbbEVT_ON = wx.NewEventType()
EVT_ON = wx.PyEventBinder(tbbEVT_ON, 1)
tbbEVT_OFF = wx.NewEventType()
EVT_OFF = wx.PyEventBinder(tbbEVT_OFF, 1)

class OnOffEvent(wx.PyCommandEvent):
    """ Event sent from the :class:`OnOff` when the control changes. """

    def __init__(self, eventType, eventId=1, value=0, size=(0,0)):
        """
        Default class constructor.

        :param `eventType`: the event type;
        :param `eventId`: the event identifier.
        """

        wx.PyCommandEvent.__init__(self, eventType, eventId)
        self._eventType = eventType
        self.value = value
        self.size = size

    def GetValue(self):
        """
        Retrieve the value of the control at the time
        this event was generated."""
        return self.value

    def GetSize(self):
        """
        Retrieve the value of the control at the time
        this event was generated."""
        return self.size

class ToggleButtonBitmap(wx.Control):

    def __init__(self, parent, id=wx.ID_ANY, bitmapon=wx.NullBitmap, bitmapoff=wx.NullBitmap, pos=wx.DefaultPosition, size=wx.DefaultSize, initial=0, style=wx.BORDER_NONE, name="OnOffButton"):
        """
        Default class constructor.

        @param parent:  Parent window. Must not be None.
        @param id:      identifier. A value of -1 indicates a default value.
        @param pos:     Position. If the position (-1, -1) is specified
                        then a default position is chosen.
        @param size:    If the default size (-1, -1) is specified then the image size is chosen.
        @param initial: Initial value 0 False or 1 True.
        @param style:   border style.
        @param name:    Widget name.
        """

        wx.Control.__init__(self, parent, id, pos=pos, size=size, style=style, name=name)

        self.parent = parent
        self._initial = initial
        self._frozen_value = initial
        self._pos = pos
        self._size = wx.Size(size)
        self._name = name
        self._id = id
        self.SetSize(self._size)
        self.SetBackgroundColour(parent.GetBackgroundColour())
        self.SetBackgroundStyle(wx.BG_STYLE_PAINT)
        if not bitmapon:
            bitmapon=wx.Bitmap(20,20)
        if not bitmapoff:
            bitmapoff=wx.Bitmap(20,20)
        self._bitmaps = {
            "On": self.SetImageSize(bitmapon),
            "Off": self.SetImageSize(bitmapoff),
            }
        self._bitmaps["OnDisabled"] = self.DisableImage(self._bitmaps["On"])
        self._bitmaps["OffDisabled"] = self.DisableImage(self._bitmaps["Off"])
        if self._initial > 1:
            self._initial = 1
        if self._initial < 0:
            self._initial = 0

        self._value = initial

        self._enabled = True
        self.InitUI()

    def InitUI(self):
        self.img = wx.StaticBitmap(self, -1, bitmap=wx.Bitmap(self.GetBitmap()))
        self.Bind(wx.EVT_SIZE, self.OnSize)
        self.img.Bind(wx.EVT_LEFT_DOWN, self.OnClick)
        self.Show()

    def OnSize(self, event):
        """
        Handles the ``wx.EVT_SIZE`` event for :class:`OnOffButton`.

        :param `event`: A `wx.SizeEvent` to be processed.
        :type `event`: `wx.SizeEvent`
        """
        self._size = self.DoGetBestSize()
        self.Refresh()

    def OnClick(self, e):
        if not self._enabled:
            return
        self._value = not self._value
        self.img.SetBitmap(self.GetBitmap())
        tp = self.GetToolTip()
        if self._value:
            tt = tp + " [On] "
        else:
            tt = tp + " [Off] "
        self.img.SetToolTip(tt)

        event = OnOffEvent(tbbEVT_ON_OFF, self.GetId(), self._value, self.img.GetSize())
        event.SetEventObject(self)
        self.GetEventHandler().ProcessEvent(event)
        if not self._value:
            #event Off
            event = OnOffEvent(tbbEVT_OFF, self.GetId(), self._value, self.img.GetSize())
            event.SetEventObject(self)
            self.GetEventHandler().ProcessEvent(event)
        else:
            #event On
            event = OnOffEvent(tbbEVT_ON, self.GetId(), self._value, self.img.GetSize())
            event.SetEventObject(self)
            self.GetEventHandler().ProcessEvent(event)
        self.Refresh()

    def GetValue(self):
        return self._value

    def SetValue(self, value):
        self._value = value
        self.Refresh()

    def Disable(self, value=True):
        self.Enable(not value)

    def Enable(self, value=True):
        self._enabled = value
        if value and self.IsEnabled(): # If value = current state do nothing
            return
        if not value and not self.IsEnabled():
            return
        wx.Control.Enable(self, value)
        bmp = self.GetBitmap()
        self.img.SetBitmap(bmp)
        tp = self.GetToolTip()
        if value:
            if self._value:
                tt = tp + " [On] "
            else:
                tt = tp + " [Off] "
        else:
            tt = tp + " [Disabled] "
        self.img.SetToolTip(tt)
        self.Refresh()

    def DisableImage(self, bmp):
        bmp = bmp.ConvertToDisabled()
        return bmp

    def SetToolTip(self, tip):
        wx.Control.SetToolTip(self, tip)
        self.Refresh()

    def GetToolTip(self):
        tp = wx.Control.GetToolTip(self)
        if not tp:
            tp = ''
        else:
            tp = tp.GetTip()
        return tp

    def IsEnabled(self):
        return wx.Control.IsEnabled(self)

    def DoGetBestSize(self):
        bitmap = self.GetBitmap()
        # Retrieve the bitmap dimensions.
        bitmapWidth, bitmapHeight = bitmap.GetWidth(), bitmap.GetHeight()
        best = wx.Size(bitmapWidth, bitmapHeight)
        self.SetSize(best)
        self.CacheBestSize(best)
        return best

    def SetForegroundColour(self, colour):
        """
        Overridden base class virtual.

        :param `colour`: Set the foreground colour of the checkboxs label.
        :type `colour`: `wx.Colour`
        """
        wx.Control.SetForegroundColour(self, colour)
        self.InitializeColours()
        self.Refresh()

    def SetBackgroundColour(self, colour):
        """
        Overridden base class virtual.

        :param `colour`: Set the background colour of the checkbox.
        :type `colour`: `wx.Colour`
        """
        wx.Control.SetBackgroundColour(self, colour)
        self.Refresh()

    def SetImageSize(self, bmp):
        bmp = wx.Bitmap(bmp)
        imgsize = bmp.GetSize()
        imgh = imgsize[1]
        imgw = imgsize[0]
        w = self._size[0]
        h = self._size[1]
        if w <= 0:
            w = imgw
        if h <= 0:
            h = imgh
        img = bmp.ConvertToImage()
        img = img.Scale(w,h,quality=wx.IMAGE_QUALITY_HIGH)
        bmp = img.ConvertToBitmap()
        return bmp

    def GetBitmap(self):
        """
        Returns the appropriate bitmap depending on state
        (On/Off/Disabled).
        """
        if not self.IsEnabled():
            if not self._value:
                return self._bitmaps["OffDisabled"]
            else:
                return self._bitmaps["OnDisabled"]
        if not self._value:
            return self._bitmaps["Off"]
        else:
            return self._bitmaps["On"]

    def SetBitmapOn(self, bitmap):
        self._bitmaps["On"] = self.SetImageSize(bitmap)
        self._bitmaps["OnDisabled"] = self.DisableImage(self._bitmaps["On"])
        self.img.SetBitmap(self.GetBitmap())
        self.Refresh()

    def SetBitmapOff(self, bitmap):
        self._bitmaps["Off"] = self.SetImageSize(bitmap)
        self._bitmaps["OffDisabled"] = self.DisableImage(self._bitmaps["Off"])
        self.img.SetBitmap(self.GetBitmap())
        self.Refresh()

if __name__ == '__main__':

    class MyFrame(wx.Frame):

        def __init__(self, parent):

            wx.Frame.__init__(self, parent, -1, "Toggle Button Bitmap Demo", size=(-1, 680))

            panel = wx.Panel(self, -1, size=(400, 500))
            panel.SetBackgroundColour('#f0ffff') # Azure
            sizer = wx.BoxSizer(wx.VERTICAL)
            self.onoff = ToggleButtonBitmap(panel, -1, bitmapon="G1.png", bitmapoff="G2.png", size=(200,200), initial=0)
            self.onoff1 = ToggleButtonBitmap(panel, -1, bitmapon="G1.png", bitmapoff="G2.png", size=(40,40), initial=1)
            self.txt = wx.TextCtrl(panel, -1, "Off", size=(50,30))
            self.onoff2 = ToggleButtonBitmap(panel, -1, bitmapon="G3.png", bitmapoff="G4.png", size=(40,20), initial=1)
            self.onoff3 = ToggleButtonBitmap(panel, -1, bitmapon="G3.png", bitmapoff="G4.png", initial=0)
            self.Bind(EVT_ON_OFF, self.OnOff, id=self.onoff.GetId())
            self.Bind(EVT_ON, self.SwOn)
            self.Bind(EVT_OFF, self.SwOff)
            sizer.Add(self.onoff, 0, wx.ALL, 20)
            sizer.Add(self.txt, 0, wx.LEFT, 90)
            sizer.Add(self.onoff1, 0, wx.ALL, 20)
            sizer.Add(self.onoff2, 0, wx.ALL, 20)
            sizer.Add(self.onoff3, 0, wx.ALL, 20)
            panel.SetSizer(sizer)
            self.onoff2.SetToolTip("Button 2")
            self.onoff2.Enable(False)
            self.timer = wx.Timer(self)
            self.Bind(wx.EVT_TIMER, self.Toggle)
            self.timer.Start(5000)


        def Toggle(self, event):
            self.onoff2.Enable(not self.onoff2.IsEnabled())

        def OnOff(self, event):
            print("Event on/off click")
            if event.GetValue():
                self.txt.SetValue("On")
            else:
                self.txt.SetValue("Off")
            event.Skip()

        def SwOn(self, event):
            obj = event.GetEventObject()
            print(obj.Name + "Event On")

        def SwOff(self, event):
            obj = event.GetEventObject()
            print(obj.Name + "Event Off")

    app = wx.App()
    frame = MyFrame(None)
    app.SetTopWindow(frame)
    frame.Show()
    app.MainLoop()
