
"""
Modified from:
    CreatingCustomControls @ http://wiki.wxpython.org/CreatingCustomControls
    by Andrea Gavana 2009-11-30 03:49:16
    
-------------------------------------------------------------------------------

The wx.PyControl has the limitation in that its built-in text label
accomodates only small-sized fonts. Also, there is no easy way (no way ?) 
to vertically center the text in the control.

By bypassing the use of the built-in label and creating a dc-written 
text bitmap, it can be placed anywhere as desired. This is exactly what
is done with the checkbox bitmap, so there is no magic necessary. The result
is that positioning both the checkbox bitmap and the text bitmap 
is trivially easy.

Since using a dc is a bit tricky, by calling the included function 
CreateTextBitmap() it is very easy to copy the returned bitmap  
on the control simply by calling dc.DrawBitmap( pos, bmap ).


Ray Pasco   
pascor@verizon.net
    2011-08-31  Initial modification to correct:
                1) Corrected dotted outline on mouse-over when control is enabled;
                2) Added proper resizing and relayout on both label and font changes.
                3) Color the entire background of the control.
"""

import wx

#------------------------------------------------------------------------------

class CustomCheckBoxMod( wx.PyControl ) :
    """
    A custom class that replicates some of the functionalities of wx.CheckBox,
    This class draws its own better looking checkbox bitmaps.
        The class constructor:

        parent      Parent container. Must not be None.
        id          CustomCheckBox identifier. 
        label       Text label displayed to the right of the checkbox.
        font        The text label's font.
        fgColor     The text label's font color.
        bgColor     Self's background color.
        pos         Self's position in self's parent's control.
        size        Self's size.
        style       Self's permanent border (outline).
        validator   Window validator
        name        Window name
        """
    
    def __init__( self, parent, id=-1, 
                  label='', 
                  font=None, 
                  fontSize=None, 
                  fgColor=None, 
                  bgColor=None, 
                  pos=wx.DefaultPosition, 
                  size=wx.DefaultSize, 
                  style=wx.NO_BORDER, 
                  validator=wx.DefaultValidator, 
                  name='CustomCheckBoxMod' ) :
        
        
        """
        Why use wx.PyControl instead of wx.Control ?
        Basically, wx.PyControl is just like its wxWidgets counterpart, except that 
          it allows some of the more common unexposed internal methods 
          to be overridden [therefore customizable] in the derived class. 
          
        Bitmaps can be placed anywhere on this control. 
        
        The CustomCheckBox functions DoGetBestSize() and AcceptsFocusFromKeyboard() 
        are overridden.
        """
        
        wx.PyControl.__init__( self, parent, id, pos, size, style, validator, name )
        
        #-----  Store the argument values..
        
        self.labelText = label
        
        self.font = wx.SystemSettings.GetFont( wx.SYS_DEFAULT_GUI_FONT )
        if font :
            self.font = font    # Replace the default with the developer's given.
        
        self.fontSize = self.font.GetPointSize()
        if fontSize :
            self.fontSize = fontSize
            self.font.SetPointSize( fontSize )
        
        self.fgColor = self.GetForegroundColour()
        if fgColor :
            try :
                self.fgColor = wx.NamedColor( fgColor )     # given in string format
            except :
                self.fgColor = fgColor                      # given in sequence format
        
        self.bgColor = self.GetBackgroundColour()
        if bgColor :
            try :
                self.bgColor = wx.NamedColor( bgColor )
            except :
                self.bgColor = bgColor
        
        #-----  Set initial states and pre-create objects used later.
        
        # Initialize the focus pen colour/dashes for faster drawing later on.
        self.Create_Focus_Pen()     # For mouse-overs, too.
        
        # By default start in the unchecked and unfocussed states.
        self.isChecked = False
        self.hasFocus  = False
        
        # Initialize our cool checkbox bitmaps.
        self.CreateCheckBoxStateBitmaps()    # Store 4 bitmap variations, all the same size.
        
        # Allow for a small border around self for a nicer appearance.
        # Value floors at 3 pixels for small fonts.
        self.textBorder = max( 3, (self.GetFont().GetPointSize() / 10) )
        
        self.SetLabel( self.labelText )     # Calls self.SetInitialSize()
        
        self.InheritAttributes()
        
        #-----  Bind only the needed events.
        
        # Bind the events related to our control.
        
        self.Bind( wx.EVT_PAINT, self.OnPaint )
        
        # Monitor user clicks so that we can switch self's state
        #   between checked and unchecked and so change the displayed bitmaps
        #   depending on the combination of current states.
        self.Bind( wx.EVT_LEFT_DOWN, self.OnMouseClick )
        
        # Avoid double-click problems on MSW.
        # Double-clicks on MSW will simply cause 2 checked state toggles on a row.
        if wx.Platform == '__WXMSW__' :
            self.Bind( wx.EVT_LEFT_DCLICK, self.OnMouseDblClick )

        # React to keyboard keys, namely the space bar that 
        #   can toggle our checked state
        self.Bind( wx.EVT_KEY_UP, self.OnKeyUp )

        # React to the focus event, because we want to draw a 
        #   dotted rectangle around self if self has focus.
        self.Bind( wx.EVT_ENTER_WINDOW, self.OnSetFocus )   # Draw dotted rectangle around self.
        self.Bind( wx.EVT_LEAVE_WINDOW, self.OnKillFocus )
        
    #end __init__
    
    #----------------------------------
    
    def Redraw( self ) :
        """ 
        Resizing and redrawing of self. This algorithm is a "hack" !
        This hack fails if the top level Frame/Dialog is maximized. 
        """
        
        # SetInitialSize() acts like "SetSize()"
        self.InvalidateBestSize()
        self.SetInitialSize( self.FindMinDimensions() )     # Sets self.minWid, self.minHgt
        
        # Force re-layout of self. HACK !  HACK !  HACK !
        # There really must be a better way, not just a different way, 
        #   to accomplish auto-resizing and auto-repainting. I wonder what that is...
        topParent = self.GetTopLevelParent()
        topWid, topHgt = topParent.GetClientSize()
        
        topParent.SetClientSize( (topWid-1, topHgt) )   # "Bump" the top level Frame size
        topParent.SetClientSize( (topWid,   topHgt) )   # Reset the size
        
        self.Refresh()      # Repaint self.
        
    #end def
    
    #----------------------------------
    
    def SetLabel( self, labelText ) :
        
        self.labelText = labelText
        self.Redraw()
        
    #----------------------------------
    
    def GetLabel( self ) :
        
        return self.labelText
    
    #----------------------------------
    
    def SetFont( self, font ) :
        
        self.font = font
        self.Redraw()
        
    #----------------------------------
    
    def SetFontSize( self, size ) :
        
        font = self.font
        font.SetPointSize( size )
        self.font = font
        self.Redraw()
        
    #----------------------------------
    
    def GetFont( self ) :
        
        return self.font
    
    #----------------------------------
    
    def FindMinDimensions( self ) :
        """
        This calculates self's best size.
        It also sets some global dimensions.
        
        """
        
        # Get the size of a checkbox bitmap.
        checkboxBmap = self.GetCheckboxBitmap()   # Temporary bitmap.
        self.checkboxBmapWid, self.checkboxBmapHgt = checkboxBmap.GetSize()
        
        # Create a disposable text label bitmap simply to measure its size.
        # This bitmap's size depends on the current font and foreground color.
        textLabelBmap = CreateTextBitmap( text=self.GetLabel(), font=self.GetFont(), 
                                          fgColor=self.fgColor, bgColor=self.bgColor, 
                                          transparent=False )

        textLabelBmapWid, textLabelBmapHgt = textLabelBmap.GetSize()
        
        # Calculate the minimum dimensions for self.
        # Set the spacing between the check bitmap and the label is fixed to 3.
        self.checkboxToTextSpacing = self.GetFont().GetPointSize() / 2      # Arbitrary spacing.
        right  = self.checkboxBmapWid + self.checkboxToTextSpacing + textLabelBmapWid + self.textBorder
        bottom = max( self.checkboxBmapHgt, textLabelBmapHgt )
        minSize = (right, bottom)
        self.minWid, self.minHgt = minSize
        
        return minSize
        
    #end FindMinDimensions def
    
    #----------------------------------
    
    def CreateCheckBoxStateBitmaps( self ) :
        """ Initializes the check bitmaps. """

        # Store 4 bitmaps of CustomCheckBox for quick access for all 
        #   state permutations of Enabled/Disabled vs. Checked/Unchecked.
        self.checkboxStateBmaps = { "CheckedEnabled"    : GetBitmap_Checked_PNG(),
                                    "UnCheckedEnabled"  : GetBitmap_NotChecked_PNG(),
                                    "CheckedDisabled"   : GrayOut( GetBitmap_Checked_PNG() ),
                                    "UnCheckedDisabled" : GrayOut( GetBitmap_NotChecked_PNG() ) }
    
    #----------------------------------
    
    def Create_Focus_Pen( self ) :
        """ 
        Create the focus indicator pen used later for mouse-over events.
        This color has to have very high color contrast with self.bgColor to be seen properly.
        """
        
        if wx.Platform == "__WXMAC__" :
            self.mouseOverIndicatorPen = wx.Pen( self.fgColor, 1, wx.SOLID )
        else :
            self.mouseOverIndicatorPen  = wx.Pen( self.fgColor, 1, wx.USER_DASH )
            self.mouseOverIndicatorPen.SetDashes( [1, 1] )     # Use a list, NOT a tuple, else BOMB !
            self.mouseOverIndicatorPen.SetCap( wx.CAP_BUTT )
        
    #----------------------------------
    
    def GetCheckboxBitmap( self ) :
        """
        Returns the appropriate checkbox bitmap 
          which depends on the current checked state and self's enabled state.
        """
        
        if self.IsEnabled() :
            # So we are Enabled
            if self.IsChecked() :
                # We are Checked
                return self.checkboxStateBmaps[ "CheckedEnabled" ]
            else :
                # We are UnChecked
                return self.checkboxStateBmaps[ "UnCheckedEnabled" ]
            #end if
        else :
            # Poor CustomCheckBox, Disabled and ignored!
            if self.IsChecked() :
                return self.checkboxStateBmaps[ "CheckedDisabled" ]
            else :
                return self.checkboxStateBmaps[ "UnCheckedDisabled" ]
            #end if
        #end if
    
    #----------------------------------
    
    def DoGetBestSize( self ) :
        """
        Overridden base class virtual.  Determines the best size of the control
        based on the checkbox and text label bitmap sizes and the spacing.
        This has already been precalculated in __init__ ().
        Calling self.SetLabel will cause self.minWid and self.minHgt
          to be recalculated, so, bestSize will always be current.
        """
        
        self.minWid, self.minHgt = self.FindMinDimensions()
        bestSize = wx.Size( self.minWid, self.minHgt )
        
        # Cache the best size so it doesn't need to be calculated again,
        # at least until some properties of the window change
        self.CacheBestSize( bestSize )
        
        return bestSize
    
    #end DoGetBestSize def
    
    #----------------------------------
    
    def AcceptsFocusFromKeyboard( self ) :
        """Overridden base class virtual."""

        # We can accept focus from keyboard, obviously
        return True


    def AcceptsFocus( self ) :
        """ Overridden base class virtual. """

        # It seems to me that wx.CheckBox does not accept focus with mouse, 
        #   but please correct me if I am wrong!
        return False        


    def HasFocus( self ) :
        """ Returns whether or not we have the focus. """

        # We just returns the _hasFocus property that has been set in the
        # wx.EVT_SET_FOCUS and wx.EVT_KILL_FOCUS event handlers.
        return self.hasFocus
    
    #----------------------------------
    
    def SetForegroundColor( self, color ) :
        """ Overridden base class virtual. """
        
        self.fgColor = color
        
        # Redraw self using the new color.
        self.Refresh()
        
    #----------------------------------
    
    def SetForegroundColour( self, colour ) :
        """ Overridden base class virtual. """
        
        self.SetForegroundColor( colour )
        
        # Redraw self using the new color.
        self.Refresh()
        
    #----------------------------------
    
    def SetBackgroundColor( self, color ) :
        """ Overridden base class virtual. """
        
        self.bgColor = color
        
        # Redraw self using the new color.
        self.Refresh()
        
    #----------------------------------
    
    def SetBackgroundColour( self, colour ) :
        """ Overridden base class virtual. """
        
        self.SetBackgroundColor( colour )
        
        # Redraw self using the new color.
        self.Refresh()
        
    #----------------------------------
    
    def Enable( self, enable=True ) :
        """ Enables/Disables CustomCheckBox. """

        wx.PyControl.Enable( self, enable )

        # Redraw self using the checkbox bitmap with its enabled color.
        self.Refresh()
    
    #----------------------------------
    
    def GetDefaultAttributes( self ) :
        """
        Overridden base class virtual.  By default we should use
        the same font/colour attributes as the native wx.CheckBox.
        """
        
        return wx.CheckBox.GetClassDefaultAttributes()
    
    #----------------------------------
    
    def ShouldInheritColours( self ) :
        """
        Overridden base class virtual.  If the parent has non-default
        colours then we want this control to inherit them.
        """
        
        return True
    
    #----------------------------------
    
    def GetSpacing( self ) :
        """ Returns the spacing between the check bitmap and the text. """
        
        return self.checkboxToTextSpacing
    
    #----------------------------------
    
    def GetValue( self ) :
        """ Returns the checked state of CustomCheckBox, True if checked. """
        
        return self.isChecked
    
    #----------------------------------
    
    def IsChecked( self ) :
        """
        This is a more intelligent synonym for GetValue(): Returns True 
          if the CustomCheckBox is checked and False otherwise.
        """
        
        return self.isChecked
    
    #----------------------------------
    
    def IsNotChecked( self ) :
        """
        This is the compliment to IsChecked(): Returns True 
          if the CustomCheckBox is checked and False otherwise.
        """
        
        return not self.isChecked
    
    #----------------------------------
    
    def SetValue( self, state ) :
        """
        Sets the CustomCheckBox to the given state. This does not cause a
        wx.wxEVT_COMMAND_CHECKBOX_CLICKED event to get emitted.
        SendCheckBoxEvent() emits the event.
        """

        self.isChecked = state

        # Refresh self: the checkbox bitmap needs to be changed.
        self.Refresh()
    
    #----------------------------------
    
    def SetChecked( self, state=True ) :
        """ This is a more intelligent synonym for SetValue(). """
        
        self.SetValue( state )
    
    #----------------------------------
    
    def SetUnchecked( self, state=True ) :
        """
        This is the complimemnt to SetChecked().
        Calling either SetUnchecked() or SetUnchecked( True ) 
           will cause self to become unchecked.
        """
        
        self.SetValue( not state )
    
    #----------------------------------
    
    def OnKeyUp( self, event ) :
        """ Handles the wx.EVT_KEY_UP event for CustomCheckBox. """

        if event.GetKeyCode() == wx.WXK_SPACE :
            
            # The spacebar has been pressed : toggle our state
            self.SendCheckBoxEvent()
            event.Skip()
            return          # Do *not* call event.Skip()
            
        else :
            event.Skip()
    
    #----------------------------------
    
    def OnSetFocus( self, event ) :
        """ Handles the wx.EVT_SET_FOCUS event for CustomCheckBox. """
        
        self.hasFocus = True
        
        # We currently have focus, so we want a dotted rectangle to be painted
        #   around the checkbox label, so we refresh ourselves.
        self.Refresh()


    def OnKillFocus( self, event ) :
        """ Handles the wx.EVT_KILL_FOCUS event for CustomCheckBox. """
        
        self.hasFocus = False

        # We lost focus, and we want a dotted rectangle to be cleared
        # around the checkbox label, so we refresh ourselves        
        self.Refresh()
    
    #----------------------------------
    
    def OnPaint( self, event ) :
        """ Handles the wx.EVT_PAINT event for CustomCheckBox. """

        # If you want to reduce flicker, a good starting point 
        #   is to use the wx.BufferedDC suitable for the particular event type.
        dc = wx.BufferedPaintDC( self )

        # Is is advisable that you don't overcrowd any OnPaint event handler 
        #   (or any other event handler) with a lot of code. 
        self.DrawSelf( dc )
    
    #----------------------------------
    
    def DrawSelf( self, dc ) :
        """
        Performs the actual drawing operations for the checkbox bitmap 
           and the text.  Both are centered vertically.
        
        All visible objects come from bitmaps and be must be drawn using a dc .
        Drawn text uses a dc to internally create and draw a bitmap of the characters.
        """
        
        self.InvalidateBestSize()
        
        # Get the actual client size of ourselves
        self.Wid, self.Hgt = self.GetClientSize()
        
        if (self.Wid < 20)  or  (self.Hgt < 20) :     # ??? Why would this ever happen ?
            # Nothing to do: Self has not yet been initially layed out.
            print '----  DrawSelf()   Self does\'nt yet have dimensions!'
            return
        
        #-----
        
        # Initialize the wx.BufferedPaintDC, assigning a background
        #   colour for self and a foreground colour for drawing the text.
        # Setting the dc brush and pen has no effect.
        dcBgColor = self.bgColor
        if not self.IsEnabled() :
            dcBgColor = MakeGray( self.bgColor, 0.70 )
        dc.SetBackground( wx.Brush( dcBgColor, wx.SOLID ) )
        dc.Clear()

        #-----  DRAW THE CHECKBOX BITMAP
        
        # Retrieve 1 of 4 checkbox bitmaps depending on the 2 state variables.
        checkboxBmap = self.GetCheckboxBitmap()
        
        # Position the checkboxBmap centered vertically within self.
        checkboxBmapPosX = 0
        checkboxBmapPosY = (self.Hgt - self.checkboxBmapHgt) / 2
        
        # Save this for mouse clicks later on.
        self.chkBoxRect = wx.RectPS( (checkboxBmapPosX, checkboxBmapPosY), checkboxBmap.GetSize() )
        
        # Draw the checkbox checkboxBmap on self using the DC tool.
        # There's no guessing where the start coordinate should be.
        dc.DrawBitmap( checkboxBmap, checkboxBmapPosX, checkboxBmapPosY, True )
        
        #-----  DRAW THE TEXT BITMAP
        
        dcTextFgColor = wx.SystemSettings.GetColour( wx.SYS_COLOUR_GRAYTEXT )
        if self.IsEnabled() :
            dcTextFgColor = self.fgColor
        
        # The code to get the dc to draw a text bitmap 
        #   is a somewhat messy, though straight-forward.
        # Calling CreateTextBitmap() makes the code here more readable.
        textLabelBmap = CreateTextBitmap( text=self.GetLabel(), font=self.GetFont(), 
                                          fgColor=dcTextFgColor, bgColor=self.bgColor, 
                                          transparent=False )
        
        # CreateTextBitmap() has already called dc.GetTextExtent()
        #   in addition to trimming the bitmap to its bestSize.
        textLabelBmapWid, textLabelBmapHgt = textLabelBmap.GetSize()
        
        # Calculate the (vertically centered) text position to avoid
        #   having to guess where its start coordinate should be 
        #   according to the particular text font.
        textLabelBmapPosX = self.checkboxBmapWid + self.checkboxToTextSpacing
        textLabelBmapPosY = (self.Hgt - textLabelBmapHgt) / 2
        dc.DrawBitmap( textLabelBmap, textLabelBmapPosX, textLabelBmapPosY, True )
        
        #-----  DRAW THE DOTTED OUTLINE ONLY IF NEEDED
        
        # Draw a dotted outline around the ENTIRE CONTROL, not just the checkbox bitmap.
        #
        # If there is focus by mouse or keyboard draw a dotted rectangle around self.
        #   This is the MS Windows behavior, I don't know on other platforms... [AG]
        #
        # Because [ self.minWid, self.minHgt ] have been precalculated, 
        #   several important capabilities are possible:
        #   1) Self's size and bestSize can be minimixed even when border padding has been included;
        #   2) Both the checkbox and the text label bitmaps can be precisely placed on self;
        
        if self.HasFocus() :
            
            # Yes, we are (self is) focused! So, use a transparent brush with
            # a dotted pen to draw a dotted rectangle on self's edges.
            
            # Find the pos and size of the bounding rectangle around self.
            #  This is both the checkboxBmap and the label text.
            dc.SetBrush( wx.TRANSPARENT_BRUSH )
            dc.SetPen( self.mouseOverIndicatorPen )
            dc.DrawRectangle( 0, 0, self.minWid, self.minHgt )
            
        #end if
    
    #end DrawSelf def
    
    #----------------------------------
    
    def OnMouseClick( self, event ) :
        """ Handles the wx.EVT_LEFT_DOWN event for CustomCheckBox. """
        
        msState = wx.GetMouseState()
        pt = self.ScreenToClient( (msState.x, msState.y) )  # Must translate the coord (literally) !
        
        # Just the area where the checkbox bitmap exists should 
        #    respond to mouse clicks, but only when self is enabled.
        if self.IsEnabled()  and  self.chkBoxRect.Inside( pt ) :
            
            self.SendCheckBoxEvent()
        
        event.Skip()
    
    #----------------------------------
    
    def OnMouseDblClick( self, event ) :
        """ 
        Avoids MSW double-click problems. 
        Interpret double-clicks as a single left-click.
        """
        pass    # Essentially ignores the second click.
    
        event.Skip()
        
    #----------------------------------
    
    def SendCheckBoxEvent( self ) :
        """ Actually sends the wx.wxEVT_COMMAND_CHECKBOX_CLICKED event. """
        
        # This part of the code may be reduced to 4 lines of code, 
        #   but it is kept as it is to help understand how it works.
        #
        # If you can, however, avoid code duplication.
        # In this case,
        # I could have written:
        #
        # self.isChecked = not self.IsChecked()
        # checkEvent = wx.CommandEvent( wx.wxEVT_COMMAND_CHECKBOX_CLICKED, self.GetId() )
        # checkEvent.SetInt( int( self.isChecked ) )
        
        if self.IsChecked() :

            self.isChecked = False      # Keep track of the new unchecked state.
            
            # Fire a wx.CommandEvent: This generates a wx.wxEVT_COMMAND_CHECKBOX_CLICKED 
            #   event that can be caught by the developer by doing something like :
            #       MyCheckBox.Bind( wx.EVT_CHECKBOX, self.OnCheckBox )
            
            checkEvent = wx.CommandEvent( wx.wxEVT_COMMAND_CHECKBOX_CLICKED, self.GetId() )
            
            checkEvent.SetInt( 0 )      # Set self to the unchecked state.

        else :

            self.isChecked = True       # Keep track of the new checked state.

            checkEvent = wx.CommandEvent( wx.wxEVT_COMMAND_CHECKBOX_CLICKED, self.GetId() )

            # Set the integer event value to 1 ( we are switching to checked state )
            checkEvent.SetInt( 1 )      # Set self to the checked state.
            
        #end if
        
        # Set the event's originating object. This is [ self ].
        checkEvent.SetEventObject( self )

        # Watch for a possible listener of this event that will catch it and
        #   eventually process it.
        self.GetEventHandler().ProcessEvent( checkEvent )

        # Refresh/repaint (our)self because the checkbox bitmap needs to be changed 
        #   to reflect the newly toggled state.
        # Redraw() doesn't need to be called because text label size hasn't changed.
        self.Refresh()
    
    #end SendCheckBoxEvent def
    
#end CustomCheckBoxMod class

#------------------------------------------------------------------------------

def GetCheckedBitmap() :
    return wx.BitmapFromImage( GetCheckedImage() )

def GetCheckedImage() :
    return wx.Image( "checked.ico", wx.BITMAP_TYPE_ICO )

def GetNotCheckedBitmap() :
    return wx.BitmapFromImage( GetNotCheckedImage() )

def GetNotCheckedImage() :
    return wx.Image( "notchecked.ico", wx.BITMAP_TYPE_ICO )

#------------------------------------------------------------------------------

def GrayOut( bmap ) :
    """
    Convert the given bitmap ( in place ) to a grayed-out version.
      This is intended for the appearance of a 'disabled' control.
    """
    
    wxImg = bmap.ConvertToImage()
    factor = 0.7    # (0.0 < f < 1.0)  Higher is more gray. 0.7 matches MSW's grayed font.
    
    if wxImg.HasMask() :
        maskColor = (wxImg.GetMaskRed(), wxImg.GetMaskGreen(), wxImg.GetMaskBlue())
    else :
        maskColor = None
    #end
    
    data = map( ord, list( wxImg.GetData() ) )

    for i in range( 0, len( data ), 3 ) :
        
        pixel = (data[i], data[i+1], data[i+2])
        pixel = MakeGray( pixel, factor, maskColor )

        for x in range( 3 ) :
            data[i+x] = pixel[x]

    wxImg.SetData( ''.join( map( chr, data ) ) )
    
    bmap = wxImg.ConvertToBitmap()
    
    return bmap
    
#end GrayOut def

#------------------------------------------------------------------------------

def MakeGray( (r, g, b), factor, maskColor=None ) :
    """
    Make a pixel appear some shade of gray. 
    If the pixel matches the maskcolor, don't gray the pixel, meaning, every 
       masked pixel stays masked and still allows the background to show through.
    
    Algorithm: Given each r, g and b value in [0..255],
               map its value non-linearly between [161..237].
    """
    newColor = (r, g, b)            # The default color (no change).
    if (r, g, b) != maskColor :     # Gray every non-mask pixel.
        newColor = map( lambda x : int( (230 - x) * factor ) + x, (r, g, b) )
    
    return newColor
    
#---------------------------------------------------------------------

# This file was generated by Image2PyFile.py

from wx.lib.embeddedimage import PyEmbeddedImage

catalog = {}
#---------------------------------------------------------------------

NotChecked_PNG = PyEmbeddedImage( 
    "iVBORw0KGgoAAAANSUhEUgAAABQAAAAUCAYAAACNiR0NAAAABHNCSVQICAgIfAhkiAAAAK9J"
    "REFUOI3N1MEKwjAMBuA/aUHw4MmhT6TmjdX38OBVfAIRPOgaD8PhGoWVRvCHwkjHR7rREHGA"
    "Z9hV+wUY88JmvdISYLvb06BAHPolIloaEdF3w3R4vRefchAD3rzBdEm+4PnkDB4PESkpmLuf"
    "9+15NBgnD98OF8sqz4LzhkAEaKtQEF5XXVsFmICkoED9/iiwS/5yXv/8LQ3Is7rrbcBmWuWB"
    "8nlYOxwMWJv/H7BPnyFe98XUuuwAAAAASUVORK5CYII="
    )

catalog["NotChecked.PNG"] = "NotChecked_PNG"
GetData_NotChecked_PNG   = NotChecked_PNG.GetData
GetImage_NotChecked_PNG  = NotChecked_PNG.GetImage
GetBitmap_NotChecked_PNG = NotChecked_PNG.GetBitmap
GetIcon_NotChecked_PNG   = NotChecked_PNG.GetIcon

#---------------------------------------------------------------------

Checked_PNG = PyEmbeddedImage( 
    "iVBORw0KGgoAAAANSUhEUgAAABQAAAAUCAYAAACNiR0NAAAABHNCSVQICAgIfAhkiAAAAThJ"
    "REFUOI2tlL1KxEAUhb+ZBPzplAUFwUYba3VAn8HSN7BzKyuLrbews1lrH2DBZh9BFKK2iiD+"
    "tNsvQWHmWqgxySXZoDkQmHty5ps7kFxjbESbsn/dKMGLBC+tAasUl43trU11allJklS/NDbK"
    "HuecNBEggIiIOOckz1AdpvJe2928na3NKuDYvNUCf/Qqj4zRWQW84LIScmQOpuYU8Ck8AzCI"
    "+gB0fa8YiKD70ctyZanPZiIpE0mzehD1i57/zeRzlR1u2HUATsIZx/YQgPOZU/BBZQDuS1dX"
    "wDWzmq2HMmLf7BVgQxmprmqBKywW6mu5YsfsZutpUsBlFlToRR6UZ7EEgvIVcIkOAY8lRggY"
    "LF8/Bng89rv2BGK9XTtzdCou00ymPA+bDIe8bm7vTC3wv2p9HrYO/ASY5pMu4l4BhwAAAABJ"
    "RU5ErkJggg=="
    )

catalog["Checked.PNG"] = "Checked_PNG"
GetData_Checked_PNG   = Checked_PNG.GetData
GetImage_Checked_PNG  = Checked_PNG.GetImage
GetBitmap_Checked_PNG = Checked_PNG.GetBitmap
GetIcon_Checked_PNG   = Checked_PNG.GetIcon

#------------------------------------------------------------------------------

def AddLinearSpacer( linearSpacingOnTheSizersAxis, boxsizer ) :
    """ 
    Insert a one-dimensional spacer for use with all BoxSizers. 
    This eliminates any possibility of a fixed-size rectangular spacer
      from interfering with adjacent controls on the sizer's minor axis.
    """
    
    orientation = boxsizer.GetOrientation()
    if   (orientation == wx.HORIZONTAL) :
        boxsizer.Add( (linearSpacingOnTheSizersAxis, 0) )
    
    elif (orientation == wx.VERTICAL) :
        boxsizer.Add( (0, linearSpacingOnTheSizersAxis) )
    
#end def

#------------------------------------------------------------------------------

def CreateTextBitmap( text, size=None, font=None, fgColor=None, bgColor=None, 
                      transparent=False ) :
    """ 
    Returns a minimally-sized dc-drawn text bitmap.
    
    Foreground and background colors may be defined as a 3-tuple or a text string.
    
    Ray Pasco
    pascor(at)verizon(dot)net
    
    2011-06-09
    2011-08-30
    """ 
    
    # The text font.
    useFont = wx.SystemSettings_GetFont( wx.SYS_DEFAULT_GUI_FONT )
    if font :
        useFont = font
        
    useSize = useFont.GetPointSize()
    if size :               # Set the font's size in pixels, not points !
        useSize = size
    useFont.SetPointSize( useSize )         # Function name is a misnomer.
                                            # Size is in pixels, not typesetting points.
    #-----
    
    # Create a bitmap much larger than the typically expected drawn text bounds
    #  in order not to inadvertently clip the text on the right and bottom edges.
    textLen = len( text )
    bmapSizeX = (useSize * textLen) * 4     # Use a bitmap much bigger than typically expected.
    bmapSizeY = useSize             * 4
    bmapSize = (bmapSizeX, bmapSizeY)          
    textBmap = wx.EmptyBitmap( *bmapSize )  # The trial or "sandbox" bitmap for dc drawing.
    
    dc = wx.MemoryDC()
    dc.SelectObject( textBmap )
    
    dc.SetFont( useFont )
    
    #-----
    
    # Text FG and BG colors.
    useFgColor = wx.BLACK     # Already the DC's default.
    if fgColor :
        useFgColor = fgColor
    dc.SetTextForeground( useFgColor )
    
    useBgColor = wx.WHITE   # Already the DC's default.
    if bgColor :
        useBgColor = bgColor
    dc.SetTextBackground( useBgColor )  # Has no effect if BG mode is wx.TRANSPARENT
    
    dc.SetBackgroundMode( wx.SOLID )    # MUST first set SOLID !  (Bug or feature ?)
    if transparent :
        dc.SetBackgroundMode( wx.TRANSPARENT )  # DC default is wx.SOLID
    
    #-----
    
    # Paint the background.
    useDcColor = wx.WHITE           # The DC's default BG color.
    if not transparent :
        useDcColor = useBgColor
        
    dc.SetBackground( wx.Brush( useDcColor ) )
    dc.Clear()
    
    #print '----  useBgColor, useFgColor, useDcColor ', useBgColor, useFgColor, useDcColor
    
    #-----
    
    dc.DrawTextPoint( text, (0, 0) )    # No need to explicitly "Blit()" from one dc to another !
    
    dc.SelectObject( wx.NullBitmap )    # Detach the DC tool from the bitmap.
    
    # Trim the bitmap down to its "best size".
    # Note:  dc.GetTextExtent() automatically positions every font's baseline 
    #   at the same Y ordinate from one text bitmap to another that use the same 
    #   particular font. This means that multiple text bitmaps using the same font 
    #   that are positioned at the same start Y ordinate will look alligned despite 
    #   that some letters such as lower case [ g, j, p and y ] may have glyphs that 
    #   extend below the font's baseline.
    textBmap = textBmap.GetSubBitmap( wx.RectPS( (0, 0), dc.GetTextExtent( text ) ) )
    
    return textBmap
    
#end CreateTextBmap def

#------------------------------------------------------------------------------

