import wx
'''
    ActiveText.py

    A drop-in replacement for wx.StaticText with added functionality.
    Envisaged originally to be used for a generic messaging area, where you want to inform the user
     about what is happening, but don't want the message left cluttering up the screen.

    It's StaticText Jim, but not as we know it!
    Not exactly singing and dancing but at least self-clearing, sliding left and right, flashing and scrolling
    Can still be used as a standard statictext, simply by using SetLabel()

    Timers for Clearing, Flashing, Cycling and Tickering
    Clearing options:   Instant, Left to Right or Right to Left
    Flasing options:    Constant, Initially or Finally
    Tickering options:  Left to Right or Right to Left, limited repetition or continuous

    A sub-classed StaticText module provides:
        self Clearing text, after a period in milliseconds
        a Flash function, to flash text every n milliseconds, with optional timeout
         both functions can be used at the same time, giving Flashing, Clearing text
        a Cycling option for text to be re-presented, a given period after it has been cleared
         Flashing is not an option with Cycling text, for re-presented text
         Cycling can be combined with directional Clearing
        a Tickering option using a fixed width font, where text runs across the screen to left or right

    Flashing can be constant, for an initial period of time or for a period time before text is cleared

    Clearing can be instant, Right to Left or Left to Right

    Tickering can repeat a fixed number of times or be continuous

    The simple News-ticker style uses a StaticText rather than a DC as used by wx.lib.ticker
    It attempts to force a tolerably stable label size, by using a fixed width font,
     although this can be overridden, results are unpredictable.
     A width should always be set with size=(width, height)i.e. (300, -1) when creating the control
     for use with the Ticker style or specify the maximum number of characters to be used for the scrolling area.
    The ticker is used to present text in a area, smaller than space allows for.

    Setting a new label (Clearing/Cycling or Tickering) will cancel any existing attributes

    Adds new functions to a standard StaticText:

        CycleStop()          Stops the cycler

        Deactivate()         Stop all timers

        GetClearPeriod()     returns Clearing period (ms)

        GetCyclePeriod()     returns cycling period (ms)

        GetFlashCountdown()  returns live calculated time before flashing starts or None

        GetFlashDelay()      returns flash delay in ms

        GetFlashPeriod()     returns period (ms)

        IsActive()           Is the control Flashing, Clearing, Cycling or Tickering returns True or False

        IsClearing()         returns True or False
                             either actively clearing or set to clear

        IsCycling()          returns True or False

        IsFlashing()         returns True or False

        IsTickering()        returns True or False

        SetClearLabel(label, period=milliseconds)
                               - Used in place of wx.StaticText.SetLabel(self,label)
                                 Sets the Label and starts the timer using the value
                                  set by SetTimer() or optional period parameter
                                 Subsequent calls override existing settings
                                 A period of < 0 turns off the clearing function
                                 A call with a period > 0 resets the timer period,
                                  which will be used by subsequent calls without a period parameter
                                 A period of None turns off the clearing function for this occurrence only
                                 (Leaves the standard SetLabel unaffected)
                                 Default label='' period = -1

        SetClearStyle(style=None, period=milliseconds)
                                Style set to None (default) when text is cleared it's instantly
                                Style = "ltr" (left to right) texted is cleared 1 character at a time
                                 from left to right
                                Style = "rtl" (right to left) texted is cleared 1 character at a time
                                 from right to left (best if using wx.ALIGN_RIGHT for the text style)
                                Period in milliseconds (default = 50)
                                 Time between each character being removed.

        SetCycleLabel(label, cycle=-1)
                               - Used in place of wx.StaticText.SetLabel(self,label)
                                 Sets the Label and starts the clearing timer using the value set by SetTimer()
                                 cycle = the amount of time to wait in ms before re-presenting the text after clearing
                                 A cycle of < 0 turns off the cycling function

        SetFlash(period=milliseconds, timeout=milliseconds)
                               - Starts/Stops the Flasher flashes every period
                                  a period of 100 = 10 times a second
                                 Call with period <= 0 to deactivate the flasher
                                  the flasher will endevour to leave the last good label showing i.e. not equal to spaces
                                 Call with period > 0 to activate the flasher using the period
                                 Call with period > 0 and timeout > 0 to activate the flasher for timeout ms
                                  a period of 250 and a timeout of 5000 will flash 4 times a second for 5 seconds
                                 Call with period > 0 and timeout < 0 to delay the flasher for timeout ms
                                  before a text Clear
                                  a negative timeout will start the flasher timeout ms before the text is cleared
                                 On Call any pre-existing Calllater for timeout cancellation or delayed flash is terminated
                                 The flasher uses GetLabel not GetLabelText
                                 Default period -1 milliseconds
                                 Default timeout None

        SetTickerLabel(label, period=milliseconds, direction='ltr', maxchars=-1, repeat=-1, facename=None)
                               - Used in place of wx.StaticText.SetLabel(self,label)
                                 Sets the Label and starts the timer
                                  period is the number of milliseconds between each tick left or right
                                  direction 'ltr' text moves Left to right
                                  direction 'rtl' text move right to left
                                  maxchars overrides any size stipulation, to provide a manual definition
                                   of how big the scrolling area is in characters
                                  repeat - the number of times to repeat the ticker label or -1 continuous
                                  facename overrides automatic selection of a fixed width font
                                   specify a specific font facename. e.g. 'Liberation Mono'
                                   Failure will drop back to automatic selection.
                                Automatically selects the first fixed width font on your system.
                                Ticker labels are constructed into a list prior to the ticker starting,
                                 so the heavy lifting is done on creation, not when running.

        SetTimer(milliseconds)
                               - The period of time before the text is cleared
                                 A value of < 1 turns off the clearing function
                                 Default -1

        TickerStop()   Stops the Ticker

        If the Flasher is running and a TimerLabel is cleared i.e. both timers were running
         at the same time, both timers are stopped, to avoid flashing a cleared text.
        A TimerLabel can be set to clear after say 10 seconds and be set to initially flash for 5 seconds for example
        A TimerLabel can be set to clear after say 10 seconds and be set to flash for 5 seconds before it clears

        Timers:
        clearer     - activates clearing    One Shot    Activates clearing
        clearing    - performs clear label  Continuous  Stops when label is clear
        flasher     - performs flashing     Continuous  Stops on Clearer activation
        cycling     - Cycles clearing text  One Shot    Repeats when clearing is finished
        tickering   - News ticker text      Continuous  Stops when told to or after the given number of repetitions

        CallLaters:
        flasher     - to stop after initial period
        flasher     - to start after wait period

        All timers Stop if you call Deactivate(), use IsActive() to test if ANY timer is running

    Events: EVT_CLEAR  event on ActiveText Clear
            Returns: Control Id, True, Control name

Author:     J Healey
Created:    20/01/2023
Copyright:  J Healey - 2023
Email:      <rolfofsaxony@gmx.com>

Written and tested on Linux Python 3.8.10  wxPython 4.2.0 gtk3 (phoenix) wxWidgets 3.2.0
Tested on Windows 11 courtesy of Ecco/ZigZag

Usage examples:

Example 1

import wx
from ActiveText import ActiveText

class Frame(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent, -1, "Self Clearing Active Text Demo")
        panel = wx.Panel(self)
        box = wx.StaticBox(panel, wx.ID_ANY, "StaticBox", pos=(50, 50), size=(300, 50))
        text1 = ActiveText(box, -1, label="", style=0)
        text1.SetTimer(10000)
        text1.SetClearLabel("This text will disappear after 10 seconds")

        text2 = ActiveText(panel, -1, label="", pos=(50, 120), size=(-1,-1), style=0)
        text2.SetTimer(15000)
        text2.SetClearLabel("This text will flash for 4 seconds\n and disappear after 15 seconds")
        text2.SetFlash(period=100, timeout=4000)

        text3 = ActiveText(panel, -1, label="", pos=(50, 170), size=(-1,-1), style=0)
        text3.SetClearStyle('ltr', 25)
        text3.SetTimer(15000)
        text3.SetClearLabel("This text will flash for 5 seconds\n before it disappears after 15 seconds")
        text3.SetFlash(period=100, timeout=-5000)

        self.Show()

app = wx.App()
frame = Frame(None)
app.MainLoop()

Example 2
Making use of the ActiveText Clear event

import wx
import ActiveText as act

class Frame(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent, -1, "Self Clearing Static Text Demo")
        box = wx.StaticBox(self, wx.ID_ANY, "StaticBox", size=(300, 50))
        text1 = act.ActiveText(box, -1, label="", name="text1")
        text1.SetTimer(10000)
        text1.SetClearLabel("This text will disappear after 10 seconds")

        text2 = act.ActiveText(self, -1, label="", name="text2")
        text2.SetTimer(15000)
        text2.SetClearLabel("This text will flash for 4 seconds\n and disappear after 15 seconds")
        text2.SetFlash(period=100, timeout=4000)

        text3 = act.ActiveText(self, -1, label="", name="text3")
        text3.SetClearStyle('ltr', 100)
        text3.SetTimer(15000)
        text3.SetClearLabel("This text will flash for 5 seconds\n before disappearing Left (slowly) after 15 seconds")
        text3.SetFlash(period=100, timeout=-5000)

        text4 = act.ActiveText(self, -1, label="", name="text4")
        text4.SetClearStyle('rtl', 25)
        text4.SetTimer(15000)
        text4.SetClearLabel("This text will disappear after 15 seconds to the Right")

        text5 = act.ActiveText(self, label="", style=wx.ALIGN_LEFT, name="ticker")
        text5.SetTickerLabel("News Ticker running x 5", 100, 'rtl', maxchars=40, repeat=5)

        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(box, 0, wx.ALL, 5)
        sizer.Add(text2, 0, wx.ALL, 5)
        sizer.Add(text3, 0, wx.ALL, 5)
        sizer.Add(text4, 0, wx.ALL, 5)
        sizer.Add(text5, 0, wx.ALL, 5)
        self.SetSizer(sizer)
        self.Bind(act.EVT_CLEAR, self.OnClear)
        self.Show()

    def OnClear(self, event):
        print(event.GetId(), event.value, event.name)

app = wx.App()
frame = Frame(None)
app.MainLoop()
'''

actEVT_CLEAR = wx.NewEventType()
EVT_CLEAR = wx.PyEventBinder(actEVT_CLEAR, 1)


class ClearEvent(wx.PyCommandEvent):
    ''' Event sent from the :class:`ActiveText` when the control is cleared.
        EVT_CLEAR  The Control value has been cleared
    '''
    def __init__(self, eventType, eventId=-1, value=True, name=''):
        '''
        Default class constructor.
        :param `eventType`: the event type;
        :param `eventId`: the event identifier.
        '''
        wx.PyCommandEvent.__init__(self, eventType, eventId)
        self._eventType = eventType
        self.value = value
        self.name = name

class ActiveText(wx.StaticText):
    def __init__(self, parent, id=wx.ID_ANY, label='', pos=wx.DefaultPosition,
                 size=wx.DefaultSize, style=0, name=''):
        wx.StaticText.__init__(self, parent, id, label, pos, size, style, name)
        #super().__init__(parent, id, label, pos, size, style, name)
        self.clearer = wx.Timer(self)
        self.clearing = wx.Timer(self)
        self.flasher = wx.Timer(self)
        self.cycle = wx.Timer(self)
        self.ticker = wx.Timer(self)
        self.delay_active = None
        self.delay_start = None
        self.clear_style = None
        self.clear_style_period = 50
        self.clear_period = -1
        self.cycle_period = -1
        self.flashtext = label
        self.BaseFont = None
        self.Bind(wx.EVT_TIMER, self.OnClear, self.clearer)
        self.Bind(wx.EVT_TIMER, self.OnClearing, self.clearing)
        self.Bind(wx.EVT_TIMER, self.OnFlash, self.flasher)
        self.Bind(wx.EVT_TIMER, self.OnCycle, self.cycle)
        self.Bind(wx.EVT_TIMER, self.OnTick, self.ticker)

    def SetTimer(self, period=-1):
        '''
      - The period of time before the text is cleared
        A value of < 1 turns off the clearing function. Default -1
        '''
        self.clear_period = period

    def SetLabel(self, label=''):
        '''
        Used in place of the normal wx.StaticText.SetLabel(label)
        purely so that a standard SetLabel can automatically Deactivate the timers like other functions
        Without this function, you would have to remember to call Deactivate separately, before calling SetLabel
        in case any of the timers are running
        '''
        self.Deactivate()
        wx.StaticText.SetLabel(self, label)
        self.original_label = label

    def SetClearLabel(self, label='', period=-1):
        '''
        Sets the Label and starts the Clearing timer using the period value
        A period of < 0 turns off the Clearing function
        A period of None turns off clearing for just this occurrence
        '''
        if self.IsActive(): #turn off timers
            self.Deactivate()
        wx.StaticText.SetLabel(self,label)
        self.original_label = label

        if not period:
            return
        if period > 0: # Reset global period
            self.clear_period = period
        if self.clear_period > 0: # Use global time-out
            self.clearer.Start(self.clear_period, wx.TIMER_ONE_SHOT)

    def SetClearStyle(self, style=None, period=50):
        '''
        Defines how text will be cleared
        Style = None (default), text is cleared instantly
        Style = "ltr" (left to right) texted is cleared 1 character at a time from left to right
        Style = "rtl" (right to left) texted is cleared 1 character at a time from right to left
        Period in milliseconds (default = 50) between each character being removed.
        '''
        self.clear_style = style
        self.clear_style_period = period

    def OnClear(self, event):
        self.SetFlash(-1)
        self.clearing.Start(self.clear_style_period, wx.TIMER_CONTINUOUS)

    def OnClearing(self, event):
        lab = self.GetLabel()
        if self.clear_style: # Clear left to right or right to left
            if self.clear_style == 'ltr':
                lab = lab[1:]
            else:
                lab = " "+lab[:-1]
        else:
            lab = ''

        wx.StaticText.SetLabel(self,lab)
        if lab.strip(): # Label not empty or all spaces yet
            return
        self.clearing.Stop()
        # event clear
        self.ClearEvent()
        wx.StaticText.SetLabel(self,'')
        if self.cycle_period > 0: # Start cycling
            self.cycle.Start(self.cycle_period, wx.TIMER_ONE_SHOT)

    def ClearEvent(self):
        event = ClearEvent(actEVT_CLEAR, self.GetId(), True, self.GetName())
        event.SetEventObject(self)
        self.GetEventHandler().ProcessEvent(event)

    def SetFlash(self, period=-1, timeout=None): # negative timeout = delayed flash for clearing text
        '''
        Default period -1 ms Default timeout None
      - Starts/Stops the Flasher flashes every period, a period of 100 = 10 times a second
        period <= 0 deactivate the flasher
        timeout > 0 activate the flasher for timeout ms
        timeout < 0 delayed activation of flasher for timeout ms before a text Clearing
        '''
        if self.delay_active: # Cancel any existing CallLater
            self.delay_active.Stop()
            self.delay_active = None
        if period < 0 and self.flasher.IsRunning():
            self.flasher.Stop()
            wx.StaticText.SetLabel(self,self.flashtext)
            return
        if self.flasher.IsRunning():
            self.flasher.Stop()
        if period < 0:
            return
        if not timeout:                                                     # Simple Start flashing
            self.flasher.Start(period, wx.TIMER_CONTINUOUS)
        elif timeout > 0:                                                   # Start flashing with flash timeout
            self.flasher.Start(period, wx.TIMER_CONTINUOUS)
            self.delay_start = None
            self.delay_active = wx.CallLater(timeout, self.SetFlash)
        elif timeout < 0:                                                   # flashing before text clear
            if self.clearer.IsRunning():
                interval = self.clearer.GetInterval()
                fstart = interval + (timeout)                               # n seconds for before clear
                if fstart > 0:
                    timeout =  abs(timeout) - interval
                    self.delay_start = wx.StopWatch()                       # Permits GetFlashCountdown
                    self.delay_start.Start(timeout)
                else:
                    self.delay_start = None
                self.delay_active = wx.CallLater(fstart,
                                               lambda per = period: self.StartReverseFlash(per)) # delayed flash

    def StartReverseFlash(self, period):
        self.delay_active = None
        self.delay_start = None
        self.flasher.Start(period, wx.TIMER_CONTINUOUS)

    def OnFlash(self, event):
        txt = self.GetLabel()
        try:
            if txt:
                self.flashtext = txt
                wx.StaticText.SetLabel(self,'')
            else:
                wx.StaticText.SetLabel(self,self.flashtext)
        except Exception as e:
            self.flasher.Stop()

    def SetCycleLabel(self, label='', cycle=-1):
        '''
        Sets the Label and starts the Clearing timer using the clear_period value
        cycle - a period in ms to wait before re-presenting the text after it has been cleared
        '''
        self.Deactivate()
        wx.StaticText.SetLabel(self,label)
        self.original_label = label
        if cycle > 0:
            self.cycle_period = cycle
        if self.clear_period > 0: # Use global time-out
            self.clearer.Start(self.clear_period, wx.TIMER_ONE_SHOT)

    def OnCycle(self, event):
        self.SetCycleLabel(self.original_label, self.cycle_period)

    def CycleStop(self):
        self.cycle.Stop()
        self.cycle_period = -1
        wx.StaticText.SetLabel(self,'')

    def SetTickerLabel(self, label='', period=-1, direction="ltr", maxchars=-1, repeat=-1, facename=None):
        if self.IsActive(): #turn off timers
            self.Deactivate()
        self.tickerlabel = label.replace('\n', ' ')
        if period < 1 or not label:
            wx.StaticText.SetLabel(self,label)
            return
        wx.StaticText.SetLabel(self,'')
        ownfont = False
        self.BaseFont = self.GetFont()
        psize = self.BaseFont.GetPointSize()
        pstyle = self.BaseFont.GetStyle()
        pweight = self.BaseFont.GetWeight()
        if facename:
            newfont = self.BaseFont
            if newfont.SetFaceName(facename):
                self.SetFont(newfont)
                ownfont = True
            else:
                ownfont = False
        if not facename or ownfont == False:
            fnts = wx.FontEnumerator()
            fnts.EnumerateFacenames()
            fnt_list = fnts.GetFacenames(fixedWidthOnly=True)   # get fixed width fonts
            if fnt_list:                                        # Grab the first one
                newfont = wx.Font(fnt_list[0])
                newfont.SetPointSize(psize)                     # adjust attrs
                newfont.SetStyle(pstyle)
                newfont.SetWeight(pweight)
                self.SetFont(newfont)
        singw = self.GetTextExtent("e").GetWidth()
        w = self.GetSize().GetWidth()
        self.maxchars = int(w / singw)
        if maxchars > 0:
            self.maxchars = maxchars
        self.OnTickerLabels(direction)
        self.ticker_repeat = repeat
        self.ticker_index = 0
        self.ticker.Start(period, wx.TIMER_CONTINUOUS)

    def TickerStop(self):
        self.ticker.Stop()
        if self.BaseFont:
            self.SetFont(self.BaseFont)
        if self.flasher.IsRunning():
            self.flasher.Stop()
        wx.StaticText.SetLabel(self,'')
        self.ClearEvent()

    def OnTickerLabels(self, direction):
        self.tickerlabels = []
        if direction == 'ltr':
            label = " "*(self.maxchars) + self.tickerlabel
            limit = 0
            for i in range (1, len(label)+1):
                lab = label[-i:]
                if i > self.maxchars:
                    limit +=1
                    lab = lab[:-limit]
                self.tickerlabels.append(lab)
        else:
            label = " "*(self.maxchars-1) + self.tickerlabel
            limit = self.maxchars
            for i in range(0,len(label)):
                lab = label[i:limit]
                limit += 1
                self.tickerlabels.append(lab)

    def OnTick(self, event):
        if not self.tickerlabels or self.ticker_repeat == 0:
            self.TickerStop()
            return
        lab = self.tickerlabels[self.ticker_index]
        wx.StaticText.SetLabel(self,lab)
        self.ticker_index +=1
        if self.ticker_index > len(self.tickerlabels) - 1:
            self.ticker_index = 0
            self.ticker_repeat -= 1

    def IsTickering(self):
        '''
        Is the control Ticker returns True or False
        '''
        return self.ticker.IsRunning()

    def IsActive(self):
        '''
        Is the control Flashing, clearing, cycling or tickering returns True or False
        '''
        if self.IsClearing() or self.IsFlashing() or self.IsCycling() or self.IsTickering():
            return True
        else:
            return False

    def IsCycling(self):
        '''
        Is the control Cyling returns True or False
        '''
        if self.cycle_period > 0:
            return True
        else:
            return False

    def IsFlashing(self):
        '''
        Is the control Flashing returns True or False
        '''
        return self.flasher.IsRunning()

    def IsClearing(self):
        '''
        Is the control Clearing returns True or False
        '''
        if self.clearer.IsRunning() or self.clearing.IsRunning():
            return True
        else:
            return False

    def Deactivate(self):
        '''
        Stop all timers
        '''
        if self.delay_active: # Cancel any existing CallLater
            self.delay_active.Stop()
            self.delay_active = None
        self.flasher.Stop()
        self.delay_start = None
        self.clearing.Stop()
        self.clearer.Stop()
        self.TickerStop()
        self.CycleStop()

    def GetFlashPeriod(self):
        '''
        Returns period between flashes
        '''
        return self.flasher.GetInterval()

    def GetFlashDelay(self):
        '''
        Returns period set before flashing starts
        '''
        if self.delay_active:
            return self.delay_active.GetInterval()
        else:
            return None

    def GetFlashCountdown(self):
        '''
        Returns live calculated time before flashing starts or None
        '''
        if self.delay_active:
            if self.delay_start:
                return abs(self.delay_start.Time())
        return None


    def GetCyclePeriod(self):
        '''
        Returns period between cycles
        '''
        return self.cycle_period

    def GetClearPeriod(self):
        '''
        Returns period before text is cleared
        '''
        return self.clear_period

# ↓↓↓↓↓ Feel free to snip this last section off, it's only the demonstration section ↓↓↓↓↓ #

class MainFrame(wx.Frame):
    def __init__(self):
        wx.Frame.__init__(self, None, -1, title='ActiveText Demonstration', size=(500, 580))
        panel = wx.Panel(self)
        self.text1 = ActiveText(panel, label="Active Text", pos=(10,10), size=(350,30), style=wx.ALIGN_LEFT, name='text1')
        self.button1 = wx.Button(panel, -1, "Self clearing Text", pos=(10,50))
        self.button2 = wx.Button(panel, -1, "Non clearing Text", pos=(10,90))
        self.button3 = wx.Button(panel, -1, "Normal Text", pos=(10,130))
        self.button4 = wx.Button(panel, -1, "Flashing On", pos=(10,170))
        self.button5 = wx.Button(panel, -1, "Flashing Off", pos=(10,210))
        self.button6 = wx.Button(panel, -1,  "Clearing Text 2", pos=(10,250))
        self.button7 = wx.Button(panel, -1,  "Flashing Clearing Text", pos=(10,290))
        self.button8 = wx.Button(panel, -1,  "Reverse Flashing Clearing Text", pos=(10,330))
        self.button9 = wx.Button(panel, -1,  "Clearing Left", pos=(10,370))
        self.button10 = wx.Button(panel, -1,  "Clearing Right", pos=(10,410))
        self.button11 = wx.Button(panel, -1,  "Cycling Text", pos=(10,450))
        self.button12 = wx.Button(panel, -1,  "Cycling Stop", pos=(130,450))
        self.button13 = wx.Button(panel, -1,  "Ticker", pos=(10,490))
        self.tdir = wx.Choice(panel, wx.ID_ANY, name="Tick dir", choices=['ltr','rtl'], pos=(130, 490))
        self.tdir.SetSelection(0)
        self.tdir.SetToolTip("Left to Right or Right to Left")
        self.period = wx.SpinCtrl(panel, -1, min=1000, max=20000, initial=5000, size=(110,-1), pos=(200,50))
        self.period.SetToolTip("Clear time period (ms)")
        self.flash_period = wx.SpinCtrl(panel, -1, min=50, max=1000, initial=100, size=(110,-1), pos=(200,170))
        self.flash_period.SetToolTip("Flash and Ticker period (ms)")
        self.flash_timeout = wx.SpinCtrl(panel, -1, min=-10000, max=10000, initial=-1, size=(110,-1), pos=(200,200))
        self.flash_timeout.SetToolTip("Flash timeout (ms) -1 = No Timeout" \
                                      "\n**** This is milliseconds ****\ne.g. Set to 3000 for 3 seconds")
        self.f_text = wx.TextCtrl(panel, wx.ID_ANY, value="Events and Timers\n", pos=(200,90), size=(280, 70),
                          style=wx.TE_MULTILINE| wx.TE_READONLY| wx.VSCROLL| wx.HSCROLL)
       #self.f_text = wx.StaticText(panel, label="", pos=(320,180))
        self.t_text = wx.StaticText(panel, label="Elapsed Time ", pos=(320,60))
        self.text9 = ActiveText(panel, label="Clear Left", pos=(320,370), name='clear1')
        self.text10 = ActiveText(panel, label="Clear Right", pos=(320,410), name='clear2')
        self.button1.Bind(wx.EVT_BUTTON, self.On_Button1)
        self.button2.Bind(wx.EVT_BUTTON, self.On_Button2)
        self.button3.Bind(wx.EVT_BUTTON, self.On_Button3)
        self.button4.Bind(wx.EVT_BUTTON, self.On_Button4)
        self.button5.Bind(wx.EVT_BUTTON, self.On_Button5)
        self.button6.Bind(wx.EVT_BUTTON, self.On_Button6)
        self.button7.Bind(wx.EVT_BUTTON, self.On_Button7)
        self.button8.Bind(wx.EVT_BUTTON, self.On_Button8)
        self.button9.Bind(wx.EVT_BUTTON, self.On_Button9)
        self.button10.Bind(wx.EVT_BUTTON, self.On_Button10)
        self.button11.Bind(wx.EVT_BUTTON, self.On_Button11)
        self.button11.Bind(wx.EVT_BUTTON, self.On_Button11)
        self.button12.Bind(wx.EVT_BUTTON, self.On_Button12)
        self.button13.Bind(wx.EVT_BUTTON, self.On_Button13)
        self.timer = wx.Timer(self)
        self.Bind(wx.EVT_TIMER, self.Flashing, self.timer)
        self.timer.Start(500)
        self.counter = 0
        self.font = wx.SystemSettings.GetFont(wx.SYS_SYSTEM_FONT)
        self.font.SetPointSize(10)
        self.text1.SetFont(self.font)
        self.text9.SetFont(self.font)
        self.text10.SetFont(self.font)
        self.text1.Bind(EVT_CLEAR, self.OnClear)
        self.text9.Bind(EVT_CLEAR, self.OnClear)
        self.text10.Bind(EVT_CLEAR, self.OnClear)
        self.Show()

    # Setting Global period
    def On_Button1(self, event):
        self.text1.SetClearStyle(None)
        period = self.period.GetValue()
        self.text1.SetTimer(period)
        period = str(period/1000)
        self.text1.SetClearLabel("This text will disappear after "+ period +" seconds")
        self.counter = 0
        self.f_text.Clear()

    # No Clearing Period acts as StaticText
    def On_Button2(self, event):
        self.text1.SetTimer(0)
        self.text1.SetClearLabel("This text is permanent")
        self.counter = 0
        self.f_text.Clear()

    # Standard StaticText
    def On_Button3(self, event):
        self.text1.SetLabel("This is a standard StaticText Label")
        self.counter = 0
        self.f_text.Clear()

    # Start Flash
    def On_Button4(self, event):
        period = self.flash_period.GetValue()
        timeout=self.flash_timeout.GetValue()
        if timeout == -1:
            timeout = None
        self.text1.SetFlash(period, timeout)
        self.counter = 0
        self.f_text.Clear()

    # Stop Flash
    def On_Button5(self, event):
        self.text1.SetFlash(-1)
        self.counter = 0
        self.f_text.Clear()

    # Setting one-off period
    def On_Button6(self, event):
        self.text1.SetClearStyle(None)
        period = self.period.GetValue()
        self.text1.SetClearLabel("This text will disappear after 3 seconds", 3000)
        self.counter = 0
        self.f_text.Clear()

    # Clearing and Flashing
    def On_Button7(self, event):
        self.text1.SetClearStyle(None)
        period = self.period.GetValue()
        self.text1.SetTimer(period)
        period = str(period/1000)
        self.text1.SetClearLabel("This text will flash for 2 secs and disappear after "+ period +" seconds")
        self.text1.SetFlash(100, timeout=2000)
        self.counter = 0
        self.f_text.Clear()

    # Clearing and Reverse Flashing
    def On_Button8(self, event):
        self.text1.SetClearStyle('ltr', 25)
        period = self.period.GetValue()
        self.text1.SetTimer(period)
        period = str(period/1000)
        self.text1.SetClearLabel("This will disappear after "+ period +" seconds,\nFlashing for the last 3 secs, clearing  Left")
        self.text1.SetFlash(100, timeout=-3000)
        self.counter = 0
        self.f_text.Clear()

    # Clearing to the Left
    def On_Button9(self, event):
        self.text9.SetClearStyle('ltr', 50)
        period = self.period.GetValue()
        self.text9.SetTimer(period)
        self.text9.SetClearLabel("Clearing Left in "+str(period/1000))
        self.counter = 0
        self.f_text.Clear()

    # Clearing to the Right
    def On_Button10(self, event):
        self.text10.SetClearStyle('rtl', 50)
        period = self.period.GetValue()
        self.text10.SetTimer(period)
        self.text10.SetClearLabel("Clearing Right in "+str(period/1000))
        self.counter = 0
        self.f_text.Clear()

    # Clearing and Cycling
    def On_Button11(self, event):
        self.text1.SetClearStyle('ltr', 50)
        period = self.period.GetValue()
        self.text1.SetTimer(period)
        self.text1.SetCycleLabel("Clearing in "+str(period/1000)+" Returning in 6", cycle=6000)
        self.counter = 0
        self.f_text.Clear()

    # Cycle Stop
    def On_Button12(self, event):
        self.text1.CycleStop()
        self.counter = 0
        self.f_text.Clear()

    # Ticker
    def On_Button13(self, event):
        tdir = self.tdir.GetStringSelection()
        if tdir == 'ltr':
            self.text1.SetTickerLabel("News Ticker running continuous",
                                      self.flash_period.GetValue(), tdir, maxchars = 40)
        else:
            self.text1.SetTickerLabel("News Ticker running repeats x 5",
                                      self.flash_period.GetValue(), tdir, maxchars = 40, repeat=5)
        self.counter = 0
        self.f_text.Clear()

    # Demonstration information only
    def Flashing(self, event):
        txt = ''
        if self.text1.IsFlashing():
            txt = "Flashing"
        else:
            txt = ""
        if self.text1.IsClearing():
            txt += " Clearing"
        if self.text1.IsCycling():
            txt += " Cycling"
        if self.text1.IsTickering():
            txt += " Tickering"
        #self.f_text.SetLabel(txt)
        if txt:
            if self.counter % 1000 == 0:
                end = " +\n"
            else:
                end = "\n"
            self.f_text.write(txt + end)
        self.counter += 500
        self.t_text.SetLabel("Elapsed time " + str(self.counter/1000))
        if self.text1.GetFlashCountdown():
            self.f_text.write("Flash delay active: " + str(self.text1.GetFlashCountdown()) + "\n")

    def OnClear(self, event):
        self.f_text.write("Cleared "+ str(event.GetId()) + " " + str(event.name) + "\n")

if __name__ == '__main__':
    app = wx.App()
    frame = MainFrame()
    app.MainLoop()
